package com.finconsgroup.itserr.marketplace.metadata.bs.dto;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;
import io.swagger.v3.oas.annotations.media.Schema;
import org.jetbrains.annotations.NotNull;
import org.springframework.core.convert.converter.Converter;
import org.springframework.stereotype.Component;

/**
 * Enumeration for filtering metadata by moderation status.
 * Only PENDING and REJECTED statuses are allowed for moderation filtering.
 */
@Schema(description = "Filter for moderation status (PENDING, APPROVED, REJECTED)")
public enum ModerationStatus {
    /**
     * Metadata pending moderation review.
     */
    PENDING("pending"),

    /**
     * Metadata that moderators have approved.
     */
    APPROVED("approved"),

    /**
     * Metadata that moderators have rejected.
     */
    REJECTED("rejected");

    private final String value;

    ModerationStatus(String value) {
        this.value = value;
    }

    @JsonValue
    public String getValue() {
        return value;
    }

    @Override
    public String toString() {
        return value;
    }

    @JsonCreator
    public static ModerationStatus fromString(String value) {
        if (value == null) {
            return null;
        }
        for (ModerationStatus filter : ModerationStatus.values()) {
            if (filter.value.equalsIgnoreCase(value)) {
                return filter;
            }
        }
        throw new IllegalArgumentException("Unknown ModerationStatus: " + value + ". Allowed values are: pending, approved, rejected");
    }

    @Component
    public static class ModerationStatusConverter implements Converter<String, ModerationStatus> {
        @Override
        public ModerationStatus convert(@NotNull String source) {
            return ModerationStatus.fromString(source);
        }
    }
}