package com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.InputModerationStatusDmDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.ModerationDmStatus;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.OutputMetadataDmDto;
import io.swagger.v3.oas.annotations.Parameter;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.data.domain.Sort;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import java.util.UUID;

/**
 * Metadata DM microservice client.
 */
public interface MetadataModerationDmClient {

    /**
     * Retrieves list of metadata for moderation.
     *
     * @param status     Filter by moderation status (PENDING or REJECTED). If not specified, returns both PENDING and REJECTED list of metadata.
     * @param pageNumber Page number for pagination (0-indexed)
     * @param pageSize   Page size for pagination
     * @param sort       Sort field name
     * @param direction  Sort direction (ASC or DESC)
     * @return A page of {@link OutputMetadataDmDto} representing list of metadata for moderation (always returns the moderation version).
     */
    @GetMapping(path = "/metadata/moderation", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputPageDto<OutputMetadataDmDto> getMetadataListForModeration(
            @RequestParam(required = false) ModerationDmStatus status,
            @RequestParam(name = "searchText", required = false) String searchText,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves a specific metadata for moderation.
     *
     * @param metadataId The unique identifier of the metadata.
     * @return An {@link OutputMetadataDmDto}
     */
    @GetMapping(path = "/metadata/{metadataId}/moderation", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputMetadataDmDto getMetadataForModeration(@PathVariable UUID metadataId);

    /**
     * Sets the moderation status of a metadata.
     *
     * @param metadataId               The unique identifier of the metadata.
     * @param inputModerationStatusDto The DTO containing the moderation status and message.
     * @return A {@link OutputMetadataDmDto} containing the complete metadata DTO and the operation type.
     */
    @PatchMapping(path = "/metadata/{metadataId}/moderation-status",
            produces = MediaType.APPLICATION_JSON_VALUE,
            consumes = MediaType.APPLICATION_JSON_VALUE)
    OutputMetadataDmDto setModerationStatus(
            @PathVariable UUID metadataId,
            @RequestBody @Valid
            @Parameter(description = "Moderation status details", required = true)
            InputModerationStatusDmDto inputModerationStatusDto
    );

}
