package com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.InputCreateMetadataDmDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.InputUpdateMetadataDmDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.MetadataCategoryDmEnum;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.MetadataDmStatus;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.OutputMetadataDmDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.OutputMetadataFieldDmDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.OutputMetadataFieldExtDmDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.client.metadatadm.dto.OutputMetadataPreviewDmDto;
import jakarta.validation.Valid;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.MediaType;
import org.springframework.lang.NonNull;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import java.util.UUID;

/**
 * Metadata DM microservice client.
 */
public interface MetadataDmClient {

    /**
     * Retrieves a paginated list of all metadata previews.
     *
     * @param name       the name of the metadata
     * @param category   the category of the metadata
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of metadata per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return ResponseEntity containing a page of OutputMetadataPreviewDto and HTTP status 200 (OK)
     */
    @GetMapping(value = "/metadata", produces = MediaType.APPLICATION_JSON_VALUE)
    Page<OutputMetadataPreviewDmDto> findAll(
            @RequestParam(required = false) String name,
            @RequestParam(required = false) MetadataCategoryDmEnum category,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves a metadata by id.
     *
     * @param metadataId the id of the metadata to retrieve
     * @return ResponseEntity containing the OutputMetadataDto and HTTP status 200 (OK)
     */
    @GetMapping(value = "/metadata/{metadataId}", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputMetadataDmDto findById(@PathVariable("metadataId") UUID metadataId);

    /**
     * Creates a new metadata.
     *
     * @param request Request.
     * @return Created metadata.
     * @throws feign.FeignException.FeignClientException.Conflict if another metadata exists with the same name
     */
    @PostMapping(path = "/metadata", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    OutputMetadataDmDto create(@Valid @RequestBody InputCreateMetadataDmDto request);

    /**
     * Updates an existing metadata.
     *
     * @param metadataId               The unique identifier of the metadata to update.
     * @param inputUpdateMetadataDmDto The DTO containing the updated information.
     * @return the updated metadata
     */
    @PutMapping(path = "/metadata/{metadataId}",
            produces = MediaType.APPLICATION_JSON_VALUE,
            consumes = MediaType.APPLICATION_JSON_VALUE)
    OutputMetadataDmDto update(@PathVariable UUID metadataId, @RequestBody @Valid InputUpdateMetadataDmDto inputUpdateMetadataDmDto);

    /**
     * Deletes a metadata.
     *
     * @param id Metadata id.
     * @return Deleted metadata.
     * @throws feign.FeignException.FeignClientException.NotFound if the metadata is not found.
     */
    @DeleteMapping(path = "/metadata/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputMetadataDmDto delete(@NonNull @PathVariable("id") UUID id);

    /**
     * Retrieves a paginated list of all metadata fields.
     *
     * @param metadataId the id of the metadata
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of metadata per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return ResponseEntity containing a page of OutputMetadataFieldDto and HTTP status 200 (OK)
     */
    @GetMapping(value = "/metadata/{metadataId}/fields", produces = MediaType.APPLICATION_JSON_VALUE)
    Page<OutputMetadataFieldDmDto> findAllFieldsById(
            @PathVariable("metadataId") UUID metadataId,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves a paginated list of all metadata fields.
     *
     * @param category   the category of the metadata
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of metadata per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return ResponseEntity containing a page of OutputMetadataPreviewDto and HTTP status 200 (OK)
     */
    @GetMapping(value = "/metadata/fields", produces = MediaType.APPLICATION_JSON_VALUE)
    Page<OutputMetadataFieldExtDmDto> findAllFields(
            @RequestParam(required = false) String name,
            @RequestParam(required = false) MetadataCategoryDmEnum category,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Submits a draft Metadata for moderation.
     *
     * @param metadataId The unique identifier of the catalog Metadata.
     * @return An {@link OutputMetadataDmDto} containing the Metadata in pending status and the operation type.
     */
    @PostMapping(path = "/metadata/{metadataId}/moderation-request", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputMetadataDmDto requestModeration(@PathVariable UUID metadataId);

    /**
     * Retrieves a paginated list of all metadata by status and user.
     *
     * @param status     Filter by moderation status
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of metadata per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return ResponseEntity containing a page of OutputMetadataDto and HTTP status 200 (OK)
     */
    @GetMapping(value = "/metadata/mine", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputPageDto<OutputMetadataDmDto> findAllByCreatorId(
            @RequestParam(required = false) MetadataDmStatus status,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );
}
