package com.finconsgroup.itserr.marketplace.metadata.bs.api;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.InputModerationStatusDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.ModerationStatus;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.data.domain.Sort;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import java.util.UUID;

/**
 * API for moderators to manage Metadata moderation.
 */
@Tag(name = "Metadata Moderation", description = "Operations for moderators to review and approve/reject list of Metadata")
@SecurityRequirement(name = "BearerAuth")
public interface MetadataModerationApi {

    /**
     * Retrieves list of metadata for moderation.
     *
     * @param status     Filter by moderation status (PENDING or REJECTED). If not specified, returns both PENDING and REJECTED list of metadata.
     * @param searchText the text to filter metadata by name and description
     * @param pageNumber Page number for pagination (0-indexed)
     * @param pageSize   Page size for pagination
     * @param sort       Sort field name
     * @param direction  Sort direction (ASC or DESC)
     * @return A page of {@link OutputMetadataDto} representing list of metadata for moderation (always returns the moderation version).
     */
    @Operation(
            summary = "Get list of metadata for moderation",
            description = "Returns list of metadata for moderation. API for moderators. Can be filtered by status (pending or rejected). By default returns both pending and rejected list of metadata."
    )
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Successfully retrieved list of metadata for moderation"),
            @ApiResponse(responseCode = "401", description = "Unauthorized", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
            @ApiResponse(responseCode = "403", description = "Forbidden - not a moderator", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
    })
    @GetMapping(path = "/metadata/moderation", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputPageDto<OutputMetadataDto>> getMetadataListForModeration(
            @RequestParam(required = false) ModerationStatus status,
            @RequestParam(name = "searchText", required = false) String searchText,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves a specific metadata for moderation.
     *
     * @param metadataId The unique identifier of the metadata.
     * @return An {@link OutputMetadataDto}
     */
    @Operation(
            summary = "Get metadata for moderation",
            description = "Returns the a metadata (any state except draft). Available to moderators."
    )
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Successfully retrieved the metadata for moderation",
                    content = @Content(schema = @Schema(implementation = OutputMetadataDto.class))),
            @ApiResponse(responseCode = "401", description = "Unauthorized", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
            @ApiResponse(responseCode = "403", description = "Forbidden - not a moderator", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
            @ApiResponse(responseCode = "404", description = "Metadata not found", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
    })
    @GetMapping(path = "/metadata/{metadataId}/moderation", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputMetadataDto> getMetadataForModeration(@PathVariable UUID metadataId);

    /**
     * Sets the moderation status of a metadata.
     *
     * @param metadataId               The unique identifier of the metadata.
     * @param inputModerationStatusDto The DTO containing the moderation status and message.
     * @return A {@link ResponseEntity} containing the complete metadata DTO and the operation type.
     */
    @Operation(
            summary = "Set moderation status",
            description = "Sets the moderation status (approved/rejected) for an metadata."
    )
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Moderation status successfully set",
                    content = @Content(schema = @Schema(implementation = OutputMetadataDto.class))),
            @ApiResponse(responseCode = "400", description = "Invalid input provided", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
            @ApiResponse(responseCode = "401", description = "Unauthorized", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
            @ApiResponse(responseCode = "403", description = "Forbidden - not a moderator", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
            @ApiResponse(responseCode = "404", description = "Metadata not found", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
    })
    @PatchMapping(path = "/metadata/{metadataId}/moderation-status",
            produces = MediaType.APPLICATION_JSON_VALUE,
            consumes = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputMetadataDto> setModerationStatus(
            @PathVariable UUID metadataId,
            @RequestBody @Valid
            @Parameter(description = "Moderation status details", required = true)
            InputModerationStatusDto inputModerationStatusDto
    );
}