package com.finconsgroup.itserr.marketplace.metadata.bs.api;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.InputCreateMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.InputUpdateMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.MetadataCategoryEnum;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.MetadataStatus;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataFieldDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataFieldExtDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataPreviewDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.UUID;

@Tag(name = "Metadata", description = "Operations related to metadata")
@SecurityRequirement(name = "BearerAuth")
public interface MetadataApi {

    /**
     * Retrieves a paginated list of all metadata previews.
     *
     * @param name       the name to search for
     * @param category   the category of the metadata
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of metadata per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return ResponseEntity containing a page of OutputMetadataPreviewDto and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieve all metadata",
            parameters = {
                    @Parameter(name = "name", description = "the name to search for"),
                    @Parameter(name = "category", description = "The category of the metadata"),
                    @Parameter(name = "pageNumber", description = "The page index, starting from 0"),
                    @Parameter(name = "pageSize", description = "The number of items per page"),
                    @Parameter(name = "sort", description = "The field by which to sort the results"),
                    @Parameter(name = "direction", description = "The sorting direction: ASC or DESC")
            },
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
            }
    )
    @ResponseStatus(HttpStatus.OK)
    @GetMapping(value = "/metadata", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputPageDto<OutputMetadataPreviewDto>> findAll(
            @RequestParam(required = false) String name,
            @RequestParam(required = false) MetadataCategoryEnum category,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves a metadata by id.
     *
     * @param metadataId the id of the metadata to retrieve
     * @return ResponseEntity containing the OutputMetadataDto and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves a metadata by id",
            responses = {
                    @ApiResponse(responseCode = "201", description = "Created",
                            content = @Content(schema = @Schema(implementation = OutputMetadataDto.class))),
                    @ApiResponse(responseCode = "404", description = "Not Found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
            }
    )
    @ResponseStatus(HttpStatus.OK)
    @GetMapping(value = "/metadata/{metadataId}", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputMetadataDto> findById(@PathVariable("metadataId") UUID metadataId);

    @Operation(
            summary = "Creates a new metadata",
            description = "Creates a new metadata.",
            responses = {
                    @ApiResponse(responseCode = "201", description = "Created",
                            content = @Content(schema = @Schema(implementation = OutputMetadataDto.class))),
                    @ApiResponse(responseCode = "400", description = "Invalid input provided", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
                    @ApiResponse(responseCode = "401", description = "Unauthorized", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
                    @ApiResponse(responseCode = "409", description = "Another metadata exists with the same name", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
                    @ApiResponse(responseCode = "503", description = "Service unavailable", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    })
            }
    )
    @ResponseStatus(HttpStatus.CREATED)
    @PostMapping(path = "/metadata", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    OutputMetadataDto create(
            @Valid @RequestBody InputCreateMetadataDto request);

    /**
     * Updates an existing metadata.
     *
     * @param metadataId             The unique identifier of the metadata to update.
     * @param inputUpdateMetadataDto The DTO containing the updated information.
     * @return the updated metadata
     */
    @Operation(
            summary = "Update an existing metadata",
            description = "Updates the details of an existing metadata identified by its ID."
    )
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "metadata successfully updated",
                    content = @Content(schema = @Schema(implementation = OutputMetadataDto.class))),
            @ApiResponse(responseCode = "400", description = "Invalid input provided", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
            @ApiResponse(responseCode = "401", description = "Unauthorized", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
            @ApiResponse(responseCode = "404", description = "metadata not found", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
    })
    @PutMapping(path = "/metadata/{metadataId}",
            produces = MediaType.APPLICATION_JSON_VALUE,
            consumes = MediaType.APPLICATION_JSON_VALUE)
    OutputMetadataDto update(
            @Parameter(description = "ID of the metadata to update", required = true)
            @PathVariable UUID metadataId,
            @RequestBody @Valid InputUpdateMetadataDto inputUpdateMetadataDto
    );

    /**
     * Deletes a metadata by id.
     *
     * @param metadataId the id of the metadata to delete
     * @return ResponseEntity containing the OutputMetadataDto and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Deletes a metadata by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Deleted",
                            content = @Content(schema = @Schema(implementation = OutputMetadataDto.class))),
                    @ApiResponse(responseCode = "404", description = "Metadata was not found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
                    @ApiResponse(responseCode = "503", description = "Service unavailable", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    })
            }
    )
    @ResponseStatus(HttpStatus.OK)
    @DeleteMapping(value = "/metadata/{metadataId}", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputMetadataDto deleteById(@PathVariable("metadataId") UUID metadataId);

    /**
     * Retrieves a paginated list of all metadata fields.
     *
     * @param metadataId the id of the metadata
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of metadata per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return ResponseEntity containing a page of OutputMetadataFieldDto and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieve all metadata fields by Id",
            parameters = {
                    @Parameter(name = "metadataId", description = "The Id of the metadata"),
                    @Parameter(name = "pageNumber", description = "The page index, starting from 0"),
                    @Parameter(name = "pageSize", description = "The number of items per page"),
                    @Parameter(name = "sort", description = "The field by which to sort the results"),
                    @Parameter(name = "direction", description = "The sorting direction: ASC or DESC")
            },
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
            }
    )
    @ResponseStatus(HttpStatus.OK)
    @GetMapping(value = "/metadata/{metadataId}/fields", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputPageDto<OutputMetadataFieldDto>> findAllFieldsById(
            @PathVariable("metadataId") UUID metadataId,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );


    /**
     * Retrieves a paginated list of all metadata fields.
     *
     * @param category   the category of the metadata
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of metadata per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return ResponseEntity containing a page of OutputMetadataPreviewDto and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieve all metadata fields for every category",
            parameters = {
                    @Parameter(name = "name", description = "The name of the metadata"),
                    @Parameter(name = "category", description = "The category of the metadata"),
                    @Parameter(name = "pageNumber", description = "The page index, starting from 0"),
                    @Parameter(name = "pageSize", description = "The number of items per page"),
                    @Parameter(name = "sort", description = "The field by which to sort the results"),
                    @Parameter(name = "direction", description = "The sorting direction: ASC or DESC")
            },
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
            }
    )
    @ResponseStatus(HttpStatus.OK)
    @GetMapping(value = "/metadata/fields", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputPageDto<OutputMetadataFieldExtDto>> findAllFields(
            @RequestParam(required = false) String name,
            @RequestParam(required = false) MetadataCategoryEnum category,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Submits a draft Metadata for moderation.
     *
     * @param metadataId The unique identifier of the catalog Metadata.
     * @return An {@link OutputMetadataDto} containing the Metadata in pending status and the operation type.
     */
    @Operation(
            summary = "Request moderation",
            description = "Submits a draft Metadata for moderation. Returns the Metadata in status pending."
    )
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Metadata successfully submitted for moderation",
                    content = @Content(schema = @Schema(implementation = OutputMetadataDto.class))),
            @ApiResponse(responseCode = "400", description = "Invalid request - Metadata not in draft status", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
            @ApiResponse(responseCode = "401", description = "Unauthorized", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
            @ApiResponse(responseCode = "403", description = "Forbidden - not the creator", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
            @ApiResponse(responseCode = "404", description = "Metadata not found", content = {
                    @Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )
            }),
    })
    @PostMapping(path = "/metadata/{metadataId}/moderation-request", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputMetadataDto> requestModeration(@PathVariable UUID metadataId);

    /**
     * Retrieves a paginated list of all metadata by status and user.
     *
     * @param status     Filter by moderation status
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of metadata per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return ResponseEntity containing a page of OutputMetadataDto and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieve all metadata by status and user",
            parameters = {
                    @Parameter(name = "status", description = "The status of the metadata"),
                    @Parameter(name = "pageNumber", description = "The page index, starting from 0"),
                    @Parameter(name = "pageSize", description = "The number of items per page"),
                    @Parameter(name = "sort", description = "The field by which to sort the results"),
                    @Parameter(name = "direction", description = "The sorting direction: ASC or DESC")
            },
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "401", description = "Unauthorized", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
            }
    )
    @ResponseStatus(HttpStatus.OK)
    @GetMapping(value = "/metadata/mine", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputPageDto<OutputMetadataDto>> findAllByCreatorId(
            @RequestParam(required = false) MetadataStatus status,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );
}
