package com.finconsgroup.itserr.marketplace.metadata.bs.service;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.InputCreateMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.InputUpdateMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.MetadataCategoryEnum;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.MetadataStatus;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataFieldDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataFieldExtDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataPreviewDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.exception.WP2ServiceUnavailableException;
import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;

import java.util.UUID;

/**
 * Metadata related service.
 */
public interface MetadataService {

    /**
     * Retrieves a paginated list of all metadata by status and user.
     *
     * @param category the category of the metadata
     * @return a page of OutputMetadataPreviewDto representing the metadata
     */
    @NonNull
    OutputPageDto<OutputMetadataPreviewDto> findAll(MetadataCategoryEnum category,
                                                    int pageNumber,
                                                    int pageSize,
                                                    String sort,
                                                    Sort.Direction direction);

    /**
     * Retrieves a metadata by id.
     *
     * @param metadataId the id of the metadata to retrieve
     * @return an OutputMetadataDto representing the metadata details
     * @throws WP2ResourceNotFoundException if no metadata is found with the given id
     */
    @NonNull
    OutputMetadataDto findById(@NonNull UUID metadataId);

    /**
     * Creates a new metadata.
     *
     * @param request Creation request. Cannot be null.
     * @return Created metadata. Never null.
     * @throws com.finconsgroup.itserr.marketplace.metadata.bs.exception.MetadataExistsException when a metadata with the same name already exists.
     * @throws WP2ServiceUnavailableException                                                    when downstream service is not available.
     */
    @NonNull
    OutputMetadataDto create(@NonNull InputCreateMetadataDto request);

    /**
     * Updates an existing metadata.
     *
     * @param metadataId             The unique identifier of the metadata to update.
     * @param inputUpdateMetadataDto The DTO containing the updated information.
     * @return the updated metadata
     */
    @NonNull
    OutputMetadataDto update(@NonNull UUID metadataId, @NonNull InputUpdateMetadataDto inputUpdateMetadataDto);

    /**
     * Deletes a metadata by id.
     *
     * @param id Metadata id.
     * @return Deleted metadata. Never null.
     * @throws com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException Metadata was not found.
     */
    @NonNull
    OutputMetadataDto deleteById(@NonNull UUID id);

    /**
     * Retrieves a paginated list of all metadata fields by Id.
     *
     * @param metadataId the id of the metadata
     * @return a page of OutputMetadataFieldDto representing the metadata
     */
    @NonNull
    OutputPageDto<OutputMetadataFieldDto> findAllFieldsById(UUID metadataId,
                                                   int pageNumber,
                                                   int pageSize,
                                                   String sort,
                                                   Sort.Direction direction);

    /**
     * Retrieves a paginated list of all metadata fields.
     *
     * @param category the category of the metadata
     * @return an OutputMetadataFieldExtDto representing the metadata field details
     * @throws WP2ResourceNotFoundException if no metadata is found with the given id
     */
    @NonNull
    OutputPageDto<OutputMetadataFieldExtDto> findAllFields(MetadataCategoryEnum category,
                                                  int pageNumber,
                                                  int pageSize,
                                                  String sort,
                                                  Sort.Direction direction);

    /**
     * Requests moderation for a specific metadata.
     *
     * @param metadataId the unique identifier of the metadata to be moderated
     * @return the moderation dto
     */
    @NonNull
    OutputMetadataDto requestModeration(@NonNull UUID metadataId);

    /**
     * Retrieves a paginated list of all metadata by status and user.
     *
     * @param status the status of metadata
     * @return a page of OutputMetadataPreviewDto representing the metadata
     */
    @NonNull
    OutputPageDto<OutputMetadataDto> findAllByCreatorId(MetadataStatus status,
                                                        int pageNumber,
                                                        int pageSize,
                                                        String sort,
                                                        Sort.Direction direction);
}
