package com.finconsgroup.itserr.marketplace.metadata.bs.component;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.metadata.bs.bean.MetadataEvent;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.enums.EventType;
import com.finconsgroup.itserr.marketplace.metadata.bs.mapper.MetadataMapper;
import com.finconsgroup.itserr.marketplace.metadata.bs.messaging.EventProducer;
import com.finconsgroup.itserr.marketplace.metadata.bs.messaging.ResourceProducer;
import com.finconsgroup.itserr.marketplace.metadata.bs.messaging.dto.MetadataMessagingAdditionalDataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.messaging.dto.MetadataMessagingModerationRequestsAdditionalDataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.messaging.dto.MetadataMessagingStatusAdditionalDataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.service.UserProfileDetailProvider;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import com.finconsgroup.itserr.messaging.dto.MessagingEventUserDto;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationListener;
import org.springframework.stereotype.Component;

import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.UUID;

/**
 * The implementation for {@link ApplicationListener} to listen to metadata events
 * and produce relevant messages.
 */
@Component
@Slf4j
public class MessagingMetadataEventListener implements ApplicationListener<MetadataEvent> {

    private final EventProducer eventProducer;
    private final ResourceProducer resourceProducer;
    private final UserProfileDetailProvider userProfileDetailProvider;
    private final MetadataMapper metadataMapper;

    public MessagingMetadataEventListener(EventProducer eventProducer,
                                          ResourceProducer resourceProducer,
                                          UserProfileDetailProvider userProfileDetailProvider,
                                          MetadataMapper metadataMapper) {
        this.eventProducer = eventProducer;
        this.resourceProducer = resourceProducer;
        this.userProfileDetailProvider = userProfileDetailProvider;
        this.metadataMapper = metadataMapper;
    }

    @Override
    public void onApplicationEvent(@NonNull MetadataEvent event) {

        MessagingEventDto<MetadataMessagingAdditionalDataDto> messagingEventDto =
                metadataMapper.toMessagingEventDto(event.getMetadata());

        MessagingEventUserDto user = MessagingEventUserDto.builder()
                .id(JwtTokenHolder.getUserId().orElse(null))
                .name(JwtTokenHolder.getName().orElse(null))
                .username(JwtTokenHolder.getPreferredUsername().orElse(null))
                .build();
        messagingEventDto.setUser(user);

        updateEventForType(event, messagingEventDto);
        publishEventForType(event, messagingEventDto);
    }

    private void updateEventForType(MetadataEvent event, MessagingEventDto<MetadataMessagingAdditionalDataDto> eventDto) {
        OutputMetadataDto outputMetadataDto = event.getMetadata();
        MetadataMessagingAdditionalDataDto additionalData = createAdditionalData(event.getEventType());
        List<String> notifyUserIds = new LinkedList<>();

        if (Set.of(EventType.CREATED, EventType.UPDATED).contains(event.getEventType())) {
            if (event.getEventType() == EventType.CREATED) {
                eventDto.setTimestamp(outputMetadataDto.getCreatedAt());
            } else {
                eventDto.setTimestamp(outputMetadataDto.getUpdatedAt());
            }
        } else if (Set.of(EventType.MODERATION_REQUESTED, EventType.DELETED).contains(event.getEventType())) {
            eventDto.setTimestamp(outputMetadataDto.getUpdatedAt());
            userProfileDetailProvider.getModeratorProfilesIds()
                    .stream()
                    .map(UUID::toString)
                    .forEach(notifyUserIds::add);
        } else if (Set.of(EventType.APPROVED, EventType.REJECTED).contains(event.getEventType())) {
            eventDto.setTimestamp(outputMetadataDto.getUpdatedAt());
            eventDto.setMessage(outputMetadataDto.getModerationMessage());
            notifyUserIds.add(outputMetadataDto.getCreator().getId().toString());
            if (additionalData instanceof MetadataMessagingStatusAdditionalDataDto statusAdditionalData) {
                statusAdditionalData.setContent(outputMetadataDto.getDescription());
                statusAdditionalData.setCreatorId(outputMetadataDto.getCreator().getId());
            }
        }

        additionalData.setNotifyUserIds(notifyUserIds);
        eventDto.setAdditionalData(additionalData);
    }

    private MetadataMessagingAdditionalDataDto createAdditionalData(EventType eventType) {
        return switch (eventType) {
            case APPROVED, REJECTED -> new MetadataMessagingStatusAdditionalDataDto();
            case MODERATION_REQUESTED -> new MetadataMessagingModerationRequestsAdditionalDataDto();
            default -> new MetadataMessagingAdditionalDataDto();
        };
    }

    private void publishEventForType(MetadataEvent event, MessagingEventDto<?> eventDto) {
        switch (event.getEventType()) {
            case CREATED -> eventProducer.publishCreateEvent(eventDto);
            case UPDATED -> eventProducer.publishUpdateEvent(eventDto);
            case MODERATION_REQUESTED -> eventProducer.publishModerationRequested(eventDto);
            case DELETED -> {
                // publish to delete resource from global search
                resourceProducer.publishDeletedResource(event.getMetadata());
                eventProducer.publishDeleteEvent(eventDto);
            }
            case APPROVED -> {
                // publish the metadata for global search only when has been really approved on D4Science
                resourceProducer.publishApprovedResource(event.getMetadata());
                eventProducer.publishStatusEvent(eventDto);
            }
            case REJECTED -> eventProducer.publishStatusEvent(eventDto);
        }
    }

}
