package com.finconsgroup.itserr.marketplace.label.dm.api;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import com.finconsgroup.itserr.marketplace.label.dm.dto.InputCreateLabelDto;
import com.finconsgroup.itserr.marketplace.label.dm.dto.OutputLabelDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.UUID;

/**
 * APIs for managing and querying the labels.
 */
@Tag(name = "Label", description = "Operations related to label management")
@SecurityRequirement(name = "BearerAuth")
@ApiResponses(value = {
    @ApiResponse(responseCode = "500", description = "Server Error",
        content = {
            @Content(
                mediaType = MediaType.APPLICATION_JSON_VALUE,
                schema = @Schema(implementation = ErrorResponseDto.class)
            )
        })
})
public interface LabelApi {

    /**
     * Creates a new label
     *
     * @param dto the input data transfer object containing label details
     * @return the created {@link OutputLabelDto} and HTTP status 200 (OK)
     */
    @Operation(
        summary = "Create new label",
        description = "Create a new label for the given context based on" +
            "the provided dto. It also performs validations and returns errors in case of validation failures"
    )
    @ApiResponses(value = {
        @ApiResponse(responseCode = "201", description = "OK"),
        @ApiResponse(responseCode = "400", description = "Bad Request",
            content = {
                @Content(
                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ErrorResponseDto.class)
                )
            })
    })
    @ResponseStatus(HttpStatus.CREATED)
    @PostMapping("/label/labels")
    OutputLabelDto create(@Valid @RequestBody InputCreateLabelDto dto);

    /**
     * Fetch label for the provided id.
     *
     * @param labelId the id of the label to fetch
     * @return the found {@link OutputLabelDto} and HTTP status 200 (Ok)
     */
    @Operation(
        summary = "Fetches label for the provided id",
        description = "Fetches a label using the provided id."
            + "It returns not found response in case if the id does not exist"
    )
    @ApiResponses(value = {
        @ApiResponse(responseCode = "200", description = "OK"),
        @ApiResponse(responseCode = "404", description = "Not Found",
            content = {
                @Content(
                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ErrorResponseDto.class)
                )
            })
    })
    @GetMapping("/label/labels/{labelId}")
    @ResponseStatus(HttpStatus.OK)
    OutputLabelDto get(@PathVariable("labelId") UUID labelId);

    /**
     * Delete label for provided id.
     *
     * @param labelId the id of the label to delete
     *                Returns HTTP status 204 (No Content), if the label is deleted successfully
     */
    @Operation(
        summary = "Deletes label for the provided id",
        description = "Deletes a label using the provided id."
            + "It returns not found response in case if the id does not exist",
        hidden = true /* For now we do not want to allow to delete labels */
    )
    @ApiResponses(value = {
        @ApiResponse(responseCode = "204", description = "No Content"),
        @ApiResponse(responseCode = "404", description = "Not Found",
            content = {
                @Content(
                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ErrorResponseDto.class)
                )
            })
    })
    @DeleteMapping("/label/labels/{labelId}")
    @ResponseStatus(HttpStatus.NO_CONTENT)
    void delete(@PathVariable("labelId") UUID labelId);

    /**
     * Retrieves a paginated list of all labels matching the provided context.
     *
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of metadata per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return ResponseEntity containing a page of OutputLabelDto and HTTP status 200 (OK)
     */
    @Operation(
        summary = "Retrieve all labels matching the provided context",
        parameters = {
            @Parameter(name = "context", description = "The context to search labels for"),
            @Parameter(name = "name", description = "The name to search labels containing it"),
            @Parameter(name = "pageNumber", description = "The page index, starting from 0"),
            @Parameter(name = "pageSize", description = "The number of items per page"),
            @Parameter(name = "sort", description = "The field by which to sort the results"),
            @Parameter(name = "direction", description = "The sorting direction: ASC or DESC")
        },
        responses = {
            @ApiResponse(responseCode = "200", description = "OK"),
        }
    )
    @ResponseStatus(HttpStatus.OK)
    @GetMapping(value = "/label/labels", produces = MediaType.APPLICATION_JSON_VALUE)
    Page<OutputLabelDto> findByContextAndName(
        @RequestParam(name = "context") String context,
        @RequestParam(name = "name", required = false) String name,
        @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
        @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
        @RequestParam(name = "sort", defaultValue = "id", required = false) String sort,
        @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

}
