package com.finconsgroup.itserr.marketplace.label.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2DuplicateResourceException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.label.dm.config.LabelProperties;
import com.finconsgroup.itserr.marketplace.label.dm.dto.InputCreateLabelDto;
import com.finconsgroup.itserr.marketplace.label.dm.dto.OutputLabelDto;
import com.finconsgroup.itserr.marketplace.label.dm.entity.LabelEntity;
import com.finconsgroup.itserr.marketplace.label.dm.mapper.LabelMapper;
import com.finconsgroup.itserr.marketplace.label.dm.repository.LabelRepository;
import com.finconsgroup.itserr.marketplace.label.dm.service.LabelService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.UUID;

/**
 * Default implementation of {@link LabelService} to perform operations related to label resources.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultLabelService implements LabelService {

    private static final String FORMAT_BUSINESS_KEY = "{name: %s, context: %s}";

    private final LabelProperties labelProperties;
    private final LabelRepository labelRepository;
    private final LabelMapper labelMapper;

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputLabelDto create(@NonNull InputCreateLabelDto dto) {
        // Checks that there is no other label with same name and context
        if (labelRepository.countByNameIgnoreCaseAndContextIgnoreCase(dto.getName(), dto.getContext()) > 0) {
            throw new WP2DuplicateResourceException(FORMAT_BUSINESS_KEY.formatted(dto.getName(), dto.getContext()));
        }
        final LabelEntity labelEntity = labelMapper.createDtoToEntity(dto);
        final LabelEntity savedLabelEntity = labelRepository.save(labelEntity);
        return labelMapper.toOutputDto(savedLabelEntity);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputLabelDto findById(@NonNull UUID labelId) {
        final OutputLabelDto outputLabelDto = labelRepository
            .findById(labelId)
            .map(labelMapper::toOutputDto)
            .orElseThrow(() -> new WP2ResourceNotFoundException(labelId));

        return outputLabelDto;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void deleteById(@NonNull UUID labelId) {
        // check if delete allowed
        if (!labelProperties.allowDelete()) {
            throw new WP2BusinessException("Delete not allowed");
        }

        // check if id exists
        if (!labelRepository.existsById(labelId)) {
            throw new WP2ResourceNotFoundException(labelId);
        }

        labelRepository.deleteById(labelId);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputLabelDto> findByContextAndName(@NonNull String context, String name, @NonNull Pageable pageable) {
        Page<LabelEntity> labelsResult;
        if (StringUtils.isNotBlank(name)) {
            labelsResult = labelRepository.findByContextIgnoreCaseAndNameStartsWithIgnoreCase(context, name, pageable);
        } else {
            labelsResult = labelRepository.findByContextIgnoreCase(context, pageable);
        }
        return labelsResult.map(labelMapper::toOutputDto);
    }
}
