package com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.ModerationStatus;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.model.ButtonModel;
import jakarta.persistence.CascadeType;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.OneToMany;
import jakarta.persistence.OrderColumn;
import jakarta.persistence.PreUpdate;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.ToString;
import lombok.experimental.SuperBuilder;
import org.hibernate.annotations.JdbcTypeCode;
import org.hibernate.type.SqlTypes;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.institutionalpage.dm.util.Constants.MAX_LONG_TEXT_LENGTH;

/**
 * Entity class representing a institutionalPage in the institutional-page service.
 * This entity is mapped to the "institutional_page" table in the database
 * and stores information such as
 * the institutionalPage name, category, creation time, and update time.
 *
 * <p>Example usage:
 * <pre>
 * InstitutionalPageEntity institutionalPageEntity = InstitutionalPageEntity.builder()
 *     .name("Criterion")
 *     .category("project")
 *     .build();
 * </pre>
 * </p>
 */
@Entity
@Table(name = "institutional_page")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class InstitutionalPageEntity extends AbstractUUIDEntity {

    /**
     * The name of the institutional page.
     */
    @Column(name = "name", nullable = false)
    private String name;

    /**
     * ID of the parent institutional page.
     */
    @Column(name = "parent_institutional_page_id")
    private UUID parentInstitutionalPageId;

    /**
     * list of institutional page ancestors.
     */
    @Column(name = "ancestor_institutional_page_ids")
    private List<UUID> ancestorInstitutionalPageIds;

    /**
     * ID of the workspace folder related to the institutional page.
     */
    @Column(name = "workspace_folder_id", nullable = false)
    private UUID workspaceFolderId;

    /**
     * The members (wp leads included) related to the institutional page.
     */
    @OneToMany(fetch = FetchType.LAZY, cascade = CascadeType.ALL, orphanRemoval = true, mappedBy = "institutionalPage")
    @OrderColumn(name = "member_order")
    @ToString.Exclude
    private List<MemberEntity> members;

    /**
     * The paragraphs related to the institutional page.
     */
    @OneToMany(fetch = FetchType.LAZY, cascade = CascadeType.ALL, orphanRemoval = true, mappedBy = "institutionalPage")
    @OrderColumn(name = "paragraph_order")
    @ToString.Exclude
    private List<ParagraphEntity> paragraphs;

    /**
     * The category of the institutional page (e.g. tool, project).
     */
    @Column(name = "category", nullable = false)
    private String category;

    /**
     * Indicates whether the institutional page is published
     */
    @Column(name = "published", nullable = false)
    private boolean published;

    /**
     * Indicates whether the institutional page should be deleted
     */
    @Column(name = "to_delete", nullable = false)
    private boolean toDelete;

    /**
     * The moderation status of the institutional page.
     */
    @Column(name = "moderation_status", nullable = false)
    private ModerationStatus moderationStatus;

    /**
     * The publication time of the institutional page.
     */
    @Column(name = "publication_time")
    private Instant publicationTime;

    /**
     * Message related to rejection of publish process.
     */
    @Column(name = "rejection_message")
    private String rejectionMessage;

    /**
     * ID of the original institutional page version.
     */
    @Column(name = "original_institutional_page_id")
    private UUID originalInstitutionalPageId;

    /**
     * ID of the updated institutional page version.
     */
    @Column(name = "updated_institutional_page_id")
    private UUID updatedInstitutionalPageId;

    /**
     * The abstract of the institutional page.
     */
    @Column(name = "abstract_content", columnDefinition = "text", nullable = false, length = MAX_LONG_TEXT_LENGTH)
    private String abstractContent;

    /**
     * Copyright text to display.
     */
    @Column(name = "copyright_text", nullable = false)
    private String copyrightText;

    /**
     * Copyright year.
     */
    @Column(name = "copyright_year", nullable = false)
    private int copyrightYear;

    /**
     * ID of the maintainer of the page.
     */
    @Column(name = "maintainer")
    private UUID maintainer;

    /**
     * list of tags.
     */
    @Column(name = "tags")
    private List<String> tags;

    /**
     * list of languages.
     */
    @Column(name = "languages")
    private List<String> languages;

    /**
     * The image URL.
     */
    @Column(name = "image_url")
    private String imageUrl;

    /**
     * The user manual URL.
     */
    @Column(name = "user_manual_url")
    private String userManualUrl;

    /**
     * The FAQs page URL.
     */
    @Column(name = "faqs_url")
    private String faqsUrl;

    /**
     * The application page version.
     */
    @Column(name = "application_version")
    private String applicationVersion;

    /**
     * The application type (e.g. Desktop Application).
     */
    @Column(name = "application_type")
    private String applicationType;

    /**
     * The License name.
     */
    @Column(name = "license")
    private String license;

    /**
     * The supported operating systems.
     */
    @Column(name = "supported_os")
    private String supportedOs;

    /**
     * URL of the source repository.
     */
    @Column(name = "repo_name")
    private String repoName;

    /**
     * URL of the website.
     */
    @Column(name = "website")
    private String website;

    /**
     * The JSON structure for extra fields.
     */
    @JdbcTypeCode(SqlTypes.JSON)
    @Column(name = "additional_fields", columnDefinition = "json")
    private LinkedHashMap<String, String> additionalFields;

    /**
     * The JSON structure for external links.
     */
    @JdbcTypeCode(SqlTypes.JSON)
    @Column(name = "external_links", columnDefinition = "json")
    private LinkedHashMap<String, String> externalLinks;

    /**
     * The JSON structure for buttons.
     */
    @JdbcTypeCode(SqlTypes.JSON)
    @Column(name = "buttons", columnDefinition = "json")
    private List<ButtonModel> buttons;

    /**
     * ID of the user who is currently updating the entity.
     */
    @Column(name = "update_locked_by")
    private UUID updateLockedBy;

    /**
     * The ID of the user who last modified the entity.
     */
    @Column(name = "last_modified_by", nullable = false)
    private UUID lastModifiedBy;

    /**
     * The timestamp when the institutionalPage was created.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    /**
     * The timestamp when the institutionalPage was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     * </p>
     */
    @Version
    private long version;

    /**
     * Sets both {@code creationTime} and {@code updateTime} before the entity is persisted.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @Override
    public void prePersist() {
        super.prePersist();
        Instant currentTime = Instant.now().truncatedTo(ChronoUnit.MICROS);
        if (creationTime == null) {
            creationTime = currentTime;
        }
        updateTime = currentTime;
    }

    /**
     * Updates {@code updateTime} just before the entity is updated.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PreUpdate
    public void onUpdate() {
        updateTime = Instant.now().truncatedTo(ChronoUnit.MICROS);
    }

}
