package com.finconsgroup.itserr.marketplace.institutionalpage.dm.service.impl;

import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.ArchiveHelper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.DtoBuilder;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.EntityBuilder;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.InstitutionalPageHelper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.MemberHelper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.MemberRequestHelper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputInviteMembersDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputPatchIPInvitationRequestDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputPatchIPJoinRequestDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputPatchMembershipDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputRemoveMembershipDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputSubmitJoinRequestDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputMembersInHierarchyDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputPendingMemberRequestDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputPendingMemberRequestsDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.InstitutionalPageEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.MemberEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.MemberRequestEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.MemberRequestStatus;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.MemberRequestType;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.CannotRemoveMaintainerException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.MembersAlreadyWpLeaderInHierarchyException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.NotInstitutionalPageWPLeaderException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.TargetUserNotMemberException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.mapper.MemberRequestMapper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.model.MembersInHierarchyModel;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.repository.MemberRequestRepository;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.service.MemberService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Default implementation of {@link MemberService}
 * to perform operations related to Members of institutionalPage
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultMemberService implements MemberService {

    private final InstitutionalPageHelper institutionalPageHelper;
    private final MemberHelper memberHelper;
    private final MemberRequestHelper memberRequestHelper;
    private final ArchiveHelper archiveHelper;

    private final MemberRequestRepository memberRequestRepository;

    private final MemberRequestMapper memberRequestMapper;

    private final EntityBuilder memberBuilder;
    private final DtoBuilder dtoBuilder;

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Boolean hasIPModerateRole() {
        return memberHelper.hasIPModerateRole();
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputPendingMemberRequestsDto findPendingMemberRequestsForInstitutionalPage(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    ) {
        // verify user is admin or wp leader or throw
        memberHelper.verifyIPModeratorOrWpLeaderOrThrow(userId, institutionalPageId);
        // retrieve institutional page only if approved
        InstitutionalPageEntity institutionalPageEntity = institutionalPageHelper
                .retrieveOriginalForMemberById(userId, institutionalPageId);
        // check if IP is root level
        boolean isRoot = institutionalPageHelper.isRoot(institutionalPageEntity);
        List<InstitutionalPageEntity> institutionalPagesInHierarchy = new LinkedList<>();
        if (isRoot) {
            // retrieve all IP ids in the hierarchy
            institutionalPagesInHierarchy.addAll(
                    institutionalPageHelper
                            .retrieveAllOriginalInHierarchyByRootId(Pageable.unpaged(), institutionalPageId)
                            .getContent()
            );
        } else {
            institutionalPagesInHierarchy.add(institutionalPageEntity);
        }
        // create Map of <institutional page id, output dto>
        Map<UUID, OutputInstitutionalPageDto> institutionPageIdToDtoMap = dtoBuilder
                .mapEntitiesToDtos(userId, institutionalPagesInHierarchy)
                .stream()
                .collect(Collectors.toMap(OutputInstitutionalPageDto::getId, Function.identity()));
        // retrieve all pending member requests for all institutional pages
        List<MemberRequestEntity> memberRequests = memberRequestRepository.findAllByInstitutionalPageIdIn(institutionPageIdToDtoMap.keySet());
        // build output dto
        List<OutputPendingMemberRequestDto> invitations = new LinkedList<>();
        List<OutputPendingMemberRequestDto> joinRequests = new LinkedList<>();
        for (MemberRequestEntity memberRequest : memberRequests) {
            // map to output dto
            OutputInstitutionalPageDto outputInstitutionalPageDto = institutionPageIdToDtoMap
                    .get(memberRequest.getInstitutionalPageId());
            OutputPendingMemberRequestDto outputPendingRequest = memberRequestMapper
                    .toOutputPendingRequestDto(memberRequest, outputInstitutionalPageDto);
            // add to invitations or joinRequests list based on memberRequestType value
            if (memberRequest.getRequestType().equals(MemberRequestType.INVITATION)) {
                invitations.add(outputPendingRequest);
            } else if (memberRequest.getRequestType().equals(MemberRequestType.JOIN_REQUEST)) {
                joinRequests.add(outputPendingRequest);
            }
        }
        // build output dto
        OutputPendingMemberRequestsDto outputPendingMemberRequestsDto = OutputPendingMemberRequestsDto.builder()
                .invitations(invitations)
                .joinRequests(joinRequests)
                .build();
        return outputPendingMemberRequestsDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputPendingMemberRequestsDto findPendingMemberRequestsForUser(
            @NonNull UUID userId
    ) {
        // retrieve all pending member requests for user
        List<MemberRequestEntity> memberRequests = memberRequestRepository.findAllByUserId(userId);
        // retrieve all institutional pages related to pending requests
        List<UUID> institutionalPageIds = memberRequests.stream()
                .map(MemberRequestEntity::getInstitutionalPageId)
                .toList();
        List<InstitutionalPageEntity> institutionalPageEntities = institutionalPageHelper
                .retrieveAllOriginalByIds(institutionalPageIds);
        Map<UUID, OutputInstitutionalPageDto> institutionPageIdToDtoMap = dtoBuilder
                .mapEntitiesToDtos(userId, institutionalPageEntities)
                .stream()
                .collect(Collectors.toMap(OutputInstitutionalPageDto::getId, Function.identity()));
        //
        List<OutputPendingMemberRequestDto> invitations = new LinkedList<>();
        List<OutputPendingMemberRequestDto> joinRequests = new LinkedList<>();
        for (MemberRequestEntity memberRequest : memberRequests) {
            // map to output dto
            UUID institutionalPageId = memberRequest.getInstitutionalPageId();
            OutputInstitutionalPageDto outputInstitutionalPageDto = institutionPageIdToDtoMap.get(institutionalPageId);
            OutputPendingMemberRequestDto outputPendingRequest = memberRequestMapper
                    .toOutputPendingRequestDto(memberRequest, outputInstitutionalPageDto);
            // add to invitations or joinRequests list based on memberRequestType value
            if (memberRequest.getRequestType().equals(MemberRequestType.INVITATION)) {
                invitations.add(outputPendingRequest);
            } else if (memberRequest.getRequestType().equals(MemberRequestType.JOIN_REQUEST)) {
                joinRequests.add(outputPendingRequest);
            }
        }
        // build output dto
        OutputPendingMemberRequestsDto outputPendingMemberRequestsDto = OutputPendingMemberRequestsDto.builder()
                .invitations(invitations)
                .joinRequests(joinRequests)
                .build();
        return outputPendingMemberRequestsDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputPendingMemberRequestsDto inviteMembers(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputInviteMembersDto inputInviteMembersDto
    ) {
        // verify user is wp leader or throw
        memberHelper.verifyWpLeaderOrThrow(userId, institutionalPageId);
        // verify invited users are not already members or throw
        memberHelper.verifyNotAlreadyMembersOrThrow(inputInviteMembersDto.getUserIds(), institutionalPageId);
        // verify invitations not exist already
        memberRequestHelper.verifyInvitationsNotAlreadyExistOrThrow(
                institutionalPageId,
                inputInviteMembersDto.getUserIds()
        );
        // for each user, create an invitation to join the specific institutional page
        boolean wpLeader = inputInviteMembersDto.getWpLeader();
        String message = inputInviteMembersDto.getMessage();
        List<MemberRequestEntity> pendingInvitations = inputInviteMembersDto.getUserIds()
                .stream()
                .map(invitedUserId ->
                        memberBuilder.buildInvitation(institutionalPageId, invitedUserId, wpLeader, message))
                .toList();
        // save invitations on db
        memberRequestRepository.saveAllAndFlush(pendingInvitations);
        // return pending requests for specific institutional page
        return findPendingMemberRequestsForInstitutionalPage(userId, institutionalPageId);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputPendingMemberRequestsDto cancelInvitation(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull UUID invitedUserId
    ) {
        // verify user is wp leader or throw
        memberHelper.verifyWpLeaderOrThrow(userId, institutionalPageId);
        // retrieve pending invitation
        MemberRequestEntity pendingInvitation = memberRequestHelper
                .retrieveInvitationOrThrow(institutionalPageId, invitedUserId);
        // archive and remove the invitation
        archiveHelper.persistArchivedCopy(pendingInvitation, MemberRequestStatus.CANCELLED, null);
        memberRequestRepository.delete(pendingInvitation);
        // return pending requests for specific institutional page
        return findPendingMemberRequestsForInstitutionalPage(userId, institutionalPageId);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputPendingMemberRequestsDto acceptOrRejectInvitation(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputPatchIPInvitationRequestDto inputPatchIPInvitationRequestDto
    ) {
        // retrieve invitation
        MemberRequestEntity invitation = memberRequestHelper.retrieveInvitationOrThrow(institutionalPageId, userId);
        if (inputPatchIPInvitationRequestDto.getApproved()) { // if invitation is approved
            // retrieve institutional page entity
            InstitutionalPageEntity institutionalPage = institutionalPageHelper
                    .retrieveOriginalById(institutionalPageId);
            if (invitation.isWpLeader()) { // if request to add user as wp leader
                // retrieve root institutional page entity
                InstitutionalPageEntity rootInstitutionalPage = institutionalPageHelper
                        .retrieveRootInstitutionalPage(institutionalPage);
                // update existing or add new member as wp leader to root institutional page
                Optional<MemberEntity> existingRootMember = rootInstitutionalPage.getMembers()
                        .stream()
                        .filter(memberEntity -> userId.equals(memberEntity.getMemberId()))
                        .findFirst();
                if (existingRootMember.isPresent()) {
                    existingRootMember.get().setWpLead(true);
                } else {
                    MemberEntity wpLeader = memberBuilder.buildWPLeader(userId, rootInstitutionalPage);
                    rootInstitutionalPage.getMembers().add(wpLeader);
                }
                // save root institutional page with updated wp leader
                institutionalPageHelper.save(userId, rootInstitutionalPage);
                // remove regular member for that user, related to all institutional pages in the entire hierarchy
                memberHelper.deleteRegularMembersFromDescendants(userId, rootInstitutionalPage.getId(), List.of(userId));
                // archive and remove the invitation
                String message = inputPatchIPInvitationRequestDto.getMessage();
                archiveHelper.persistArchivedCopy(invitation, MemberRequestStatus.APPROVED, message);
                memberRequestRepository.delete(invitation);
                // also delete existing invitations or join requests for same user for all
                // descendant institutional page in the entire hierarchy, if any
                String cancelMessage = "user added as wpLeader";
                List<MemberRequestEntity> memberRequests = memberRequestRepository
                        .findAllRequestsFromIpAndDescendants(rootInstitutionalPage.getId(), List.of(userId));
                archiveHelper.persistArchivedCopy(memberRequests, MemberRequestStatus.CANCELLED, cancelMessage);
                memberRequestRepository.deleteAll(memberRequests);
            } else { // request to add user as regular member
                // add new regular member to institutional page
                MemberEntity member = memberBuilder.buildRegularMember(userId, institutionalPageId);
                institutionalPage.getMembers().add(member);
                // save institutional page with updated member
                institutionalPageHelper.save(userId, institutionalPage);
                // archive and remove the invitation
                String message = inputPatchIPInvitationRequestDto.getMessage();
                archiveHelper.persistArchivedCopy(invitation, MemberRequestStatus.APPROVED, message);
                memberRequestRepository.delete(invitation);
                // also delete existing join request for same user and institutional page, if any
                memberRequestHelper.retrieveJoinRequest(institutionalPageId, userId)
                        .ifPresent(joinRequest -> {
                            // archive and remove the join request
                            String cancelMessage = "accepted invitation";
                            archiveHelper.persistArchivedCopy(joinRequest, MemberRequestStatus.CANCELLED, cancelMessage);
                            memberRequestRepository.delete(joinRequest);
                        });
            }
        } else { // invitation not approved
            // archive and remove the invitation
            String message = inputPatchIPInvitationRequestDto.getMessage();
            archiveHelper.persistArchivedCopy(invitation, MemberRequestStatus.REJECTED, message);
            memberRequestRepository.delete(invitation);
        }
        // return pending requests for user
        return findPendingMemberRequestsForUser(userId);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputInstitutionalPageDto patchMembershipOfUsersForInstitutionalPage(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputPatchMembershipDto inputPatchMembershipDto
    ) {
        UUID targetUserId = inputPatchMembershipDto.getUserId();
        boolean promoteToWpLeader = inputPatchMembershipDto.getWpLeader();
        // verify user is wp leader or throw
        memberHelper.verifyWpLeaderOrThrow(userId, institutionalPageId);
        // retrieve institutional page entity
        InstitutionalPageEntity institutionalPage = institutionalPageHelper
                .retrieveOriginalForMemberById(userId, institutionalPageId);
        // verify target user is member of input institutional page or throw
        if (!memberHelper.isMember(targetUserId, institutionalPage)) {
            throw new TargetUserNotMemberException(targetUserId, institutionalPageId);
        }
        // retrieve root institutional page entity
        InstitutionalPageEntity rootInstitutionalPage = institutionalPageHelper
                .retrieveRootForMemberById(userId, institutionalPageId);
        if (rootInstitutionalPage.getMembers() == null) {
            rootInstitutionalPage.setMembers(new LinkedList<>());
        }
        // retrieve member, if exists
        MemberEntity member = rootInstitutionalPage.getMembers()
                .stream()
                .filter(memberEntity -> memberEntity.getMemberId().equals(targetUserId))
                .findFirst()
                .orElse(null);
        if (promoteToWpLeader) { // request to make target user a wp leader
            if (member != null) { // target user already member of root institutional page
                if (member.isWpLead()) { // target user already wp leader
                    throw new MembersAlreadyWpLeaderInHierarchyException(institutionalPageId, Set.of(targetUserId));
                } else { // target user is just regular member, set wpLeader to 'true'
                    member.setWpLead(true);
                }
            } else { // target user NOT member of root institutional page
                // add new wp leader member to root institutional page
                MemberEntity wpLeader = memberBuilder.buildWPLeader(targetUserId, rootInstitutionalPage);
                rootInstitutionalPage.getMembers().add(wpLeader);
            }
            // remove regular member for that user, related to all institutional pages in the entire hierarchy
            memberHelper.deleteRegularMembersFromDescendants(userId, rootInstitutionalPage.getId(), List.of(targetUserId));
            // also delete existing invitations or join requests for same users for all
            // descendant institutional page in the entire hierarchy, if any
            String message = "user promoted to wpLeader";
            List<MemberRequestEntity> memberRequests = memberRequestRepository
                    .findAllRequestsFromIpAndDescendants(rootInstitutionalPage.getId(), List.of(targetUserId));
            archiveHelper.persistArchivedCopy(memberRequests, MemberRequestStatus.CANCELLED, message);
            memberRequestRepository.deleteAll(memberRequests);
        } else { // request to make target user a regular member
            if (member != null && member.isWpLead()) { // target user is wp leader, set wpLeader to 'false'
                member.setWpLead(false);
            } else { // target user already regular member
                throw new NotInstitutionalPageWPLeaderException(institutionalPageId, targetUserId);
            }
        }
        // save updated members
        InstitutionalPageEntity savedInstitutionalPage = institutionalPageHelper.save(userId, rootInstitutionalPage);
        // return institutional page
        OutputInstitutionalPageDto outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, savedInstitutionalPage);
        return outputInstitutionalPageDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputInstitutionalPageDto removeMembershipOfUsersForInstitutionalPage(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputRemoveMembershipDto inputRemoveMembershipDto
    ) {
        UUID targetUserId = inputRemoveMembershipDto.getUserId();
        if (userId.equals(targetUserId)) { // true if user is trying to remove himself from institutional page
            // verify user is member of input institutional page or throw
            memberHelper.verifyMemberOrThrow(userId, institutionalPageId);
        } else { // user is trying to remove other users from institutional page
            // verify user is wp leader or throw
            memberHelper.verifyWpLeaderOrThrow(userId, institutionalPageId);
        }
        // retrieve institutional page entity
        InstitutionalPageEntity institutionalPage = institutionalPageHelper
                .retrieveOriginalForMemberById(userId, institutionalPageId);
        // the output dto
        OutputInstitutionalPageDto outputInstitutionalPageDto;
        //
        if (memberHelper.isWpLeader(targetUserId, institutionalPage)) {
            // In this case the 'userId' is also a wp leader because of the previous checks.
            // retrieve root institutional page
            InstitutionalPageEntity rootInstitutionalPage = institutionalPageHelper
                    .retrieveRootForMemberById(userId, institutionalPageId);
            //
            verifyUserIsNotMaintainerOfAnyIPInTheHierarchyOrThrow(targetUserId, rootInstitutionalPage);
            // remove member from root institutional page
            rootInstitutionalPage.getMembers()
                    .removeIf(memberEntity -> memberEntity.getMemberId().equals(targetUserId));
            // save institutional page with removed member
            institutionalPageHelper.save(userId, rootInstitutionalPage);
            // return the initial institutional page, not the saved root
            outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, institutionalPage);
        } else if (memberHelper.isRegularMember(targetUserId, institutionalPage)) {
            // remove member from given institutional page
            institutionalPage.getMembers()
                    .removeIf(memberEntity -> memberEntity.getMemberId().equals(targetUserId));
            // save institutional page with removed member
            InstitutionalPageEntity savedInstitutionalPage = institutionalPageHelper.save(userId, institutionalPage);
            // return the saved institutional page
            outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, savedInstitutionalPage);
        } else {
            throw new TargetUserNotMemberException(targetUserId, institutionalPageId);
        }
        // return output dto
        return outputInstitutionalPageDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputPendingMemberRequestsDto submitJoinRequest(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputSubmitJoinRequestDto inputSubmitJoinRequestDto
    ) {
        // verify not already member or throw
        memberHelper.verifyNotAlreadyMemberOrThrow(userId, institutionalPageId);
        // user can join Published institutional pages only
        institutionalPageHelper.retrievePublishedById(institutionalPageId);
        // verify join request not exist already
        memberRequestHelper.verifyJoinRequestsNotAlreadyExistOrThrow(institutionalPageId, List.of(userId));
        // create join request
        String message = inputSubmitJoinRequestDto.getMessage();
        MemberRequestEntity joinRequest = memberBuilder.buildJoinRequest(institutionalPageId, userId, message);
        // save join request
        memberRequestRepository.saveAndFlush(joinRequest);
        // return pending requests for user
        return findPendingMemberRequestsForUser(userId);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputPendingMemberRequestsDto cancelJoinRequest(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    ) {
        // retrieve pending join request
        MemberRequestEntity joinRequest = memberRequestHelper.retrieveJoinRequestOrThrow(institutionalPageId, userId);
        // archive and remove the join request
        archiveHelper.persistArchivedCopy(joinRequest, MemberRequestStatus.CANCELLED, null);
        memberRequestRepository.delete(joinRequest);
        // return pending requests for user
        return findPendingMemberRequestsForUser(userId);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputPendingMemberRequestsDto acceptOrRejectJoinRequests(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputPatchIPJoinRequestDto inputPatchIPJoinRequestDto
    ) {
        // verify user is wp leader or throw
        memberHelper.verifyWpLeaderOrThrow(userId, institutionalPageId);
        // retrieve join requests for all users, throw exception if some requests don't exist
        List<MemberRequestEntity> joinRequests = memberRequestHelper.retrieveAllJoinRequestsByUserIdsOrThrow(
                institutionalPageId,
                inputPatchIPJoinRequestDto.getUserIds()
        );
        // if request is approved
        if (inputPatchIPJoinRequestDto.getApproved()) {
            // retrieve institutional page entity
            InstitutionalPageEntity institutionalPageEntity = institutionalPageHelper
                    .retrieveOriginalForMemberById(userId, institutionalPageId);
            // for each user, save a member entity for the specific institutional page
            if (institutionalPageEntity.getMembers() == null) {
                institutionalPageEntity.setMembers(new LinkedList<>());
            }
            joinRequests.stream()
                    .map(joinRequest -> memberBuilder
                            .buildRegularMember(joinRequest.getUserId(), institutionalPageId)
                    )
                    .forEach(member -> institutionalPageEntity.getMembers().add(member));
            institutionalPageHelper.save(userId, institutionalPageEntity);
            // also delete existing invitations for same users and institutional page, if any
            String message = "accepted join request";
            List<MemberRequestEntity> invitations = memberRequestHelper
                    .retrieveInvitations(institutionalPageId, inputPatchIPJoinRequestDto.getUserIds());
            archiveHelper.persistArchivedCopy(invitations, MemberRequestStatus.CANCELLED, message);
            memberRequestRepository.deleteAll(invitations);
        }
        // archive and remove the join request
        MemberRequestStatus memberRequestStatus = inputPatchIPJoinRequestDto.getApproved()
                ? MemberRequestStatus.APPROVED
                : MemberRequestStatus.REJECTED;
        String message = inputPatchIPJoinRequestDto.getMessage();
        archiveHelper.persistArchivedCopy(joinRequests, memberRequestStatus, message);
        memberRequestRepository.deleteAll(joinRequests);
        // return pending requests for specific institutional page
        return findPendingMemberRequestsForInstitutionalPage(userId, institutionalPageId);
    }

    // private

    @NonNull
    @Override
    // @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputMembersInHierarchyDto> findAllUsersInHierarchy(
            @NonNull UUID userId,
            @NonNull Pageable pageable,
            @NonNull UUID rootInstitutionalPageId
    ) {
        // verify user is wp leader or throw
        memberHelper.verifyWpLeaderOrThrow(userId, rootInstitutionalPageId);
        // retrieve all IP ids in the hierarchy
        List<InstitutionalPageEntity> institutionalPagesInHierarchy = institutionalPageHelper
                .retrieveAllOriginalInHierarchyByRootId(Pageable.unpaged(), rootInstitutionalPageId)
                .getContent();
        // create Map of <institutional page id, output dto>
        Map<UUID, OutputInstitutionalPageDto> institutionPageIdToDtoMap = dtoBuilder
                .mapEntitiesToDtos(userId, institutionalPagesInHierarchy)
                .stream()
                .collect(Collectors.toMap(OutputInstitutionalPageDto::getId, Function.identity()));
        // retrieve all members in hierarchy by root
        MembersInHierarchyModel membersInHierarchy = memberHelper.findUsersByRootId(pageable, rootInstitutionalPageId);
        List<OutputMembersInHierarchyDto> OutputMembersInHierarchyDtos = membersInHierarchy.getMembers()
                .stream()
                .collect(Collectors.groupingBy(MemberEntity::getMemberId))
                .entrySet()
                .stream()
                .map(entry -> DtoBuilder.toMembersInHierarchyDto(
                        rootInstitutionalPageId,
                        entry,
                        institutionPageIdToDtoMap
                ))
                .toList();
        // return page of output dtos
        return new PageImpl<>(
                OutputMembersInHierarchyDtos,
                membersInHierarchy.getPageable(),
                membersInHierarchy.getTotalElements()
        );
    }

    private void verifyUserIsNotMaintainerOfAnyIPInTheHierarchyOrThrow(
            UUID targetUserId,
            InstitutionalPageEntity rootInstitutionalPage
    ) {
        Set<UUID> maintainedInstitutionalPageIds = institutionalPageHelper
                .retrieveAllOriginalInHierarchyByRootId(Pageable.unpaged(), rootInstitutionalPage.getId())
                .getContent()
                .stream()
                .filter(institutionalPageEntity -> targetUserId.equals(institutionalPageEntity.getMaintainer()))
                .map(InstitutionalPageEntity::getId)
                .collect(Collectors.toSet());
        if (!maintainedInstitutionalPageIds.isEmpty()) {
            throw new CannotRemoveMaintainerException(targetUserId, maintainedInstitutionalPageIds);
        }
    }

}