package com.finconsgroup.itserr.marketplace.institutionalpage.dm.component;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.InstitutionalPageEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.InstitutionalPageView;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.ModerationStatus;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.InstitutionalPageNotFoundException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.mapper.InstitutionalPageMapper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.model.SearchForMemberInstitutionalPageModel;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.model.SearchInstitutionalPageModel;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.repository.InstitutionalPageRepository;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.util.SpringDataUtils;
import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * Helper component that encapsulates common logic for retrieving, saving, and deleting
 * {@link InstitutionalPageEntity} instances with various filtering and moderation criteria.
 *
 * <p>
 * Provides utility methods to navigate hierarchical relationships, handle moderation workflows,
 * resolve original and root versions, and ensure data consistency through validation and exception handling.
 * </p>
 */
@Component
@RequiredArgsConstructor
public class InstitutionalPageHelper {

    private final InstitutionalPageRepository institutionalPageRepository;
    private final MemberRequestHelper memberRequestHelper;
    private final InstitutionalPageMapper mapper;

    //

    public boolean isRoot(InstitutionalPageEntity institutionalPageEntity) {
        return institutionalPageEntity.getParentInstitutionalPageId() == null;
    }

    // save and delete

    public InstitutionalPageEntity save(UUID userId, InstitutionalPageEntity institutionalPageEntity) {
        institutionalPageEntity.setLastModifiedBy(userId);
        return institutionalPageRepository.saveAndFlush(institutionalPageEntity);
    }

    public void saveAll(UUID userId, List<InstitutionalPageEntity> institutionalPageEntities) {
        institutionalPageEntities.forEach(institutionalPageEntity ->
                institutionalPageEntity.setLastModifiedBy(userId)
        );
        institutionalPageRepository.saveAllAndFlush(institutionalPageEntities);
    }

    public void delete(InstitutionalPageEntity institutionalPageEntity) {
        memberRequestHelper.deleteMemberRequestsForInstitutionalPage(institutionalPageEntity.getId());
        institutionalPageRepository.delete(institutionalPageEntity);
    }

    // retrieve by id

    public InstitutionalPageEntity retrieveInstitutionalPageOrThrow(
            UUID institutionalPageId,
            Supplier<? extends RuntimeException> exceptionSupplier
    ) {
        return institutionalPageRepository.findById(institutionalPageId).orElseThrow(exceptionSupplier);
    }

    public InstitutionalPageEntity retrieveInstitutionalPage(UUID institutionalPageId) {
        return retrieveInstitutionalPageOrThrow(
                institutionalPageId,
                () -> new InstitutionalPageNotFoundException(institutionalPageId)
        );
    }

    // latest by id

    public InstitutionalPageEntity retrieveLatestById(UUID institutionalPageId) {
        SearchInstitutionalPageModel searchFilters = SearchInstitutionalPageModel.builder()
                .ids(List.of(institutionalPageId))
                .build();
        return retrieveAllWithFilter(PageRequest.of(0, 1), searchFilters, null, false, true)
                .stream()
                .findFirst()
                .orElseThrow(() -> new InstitutionalPageNotFoundException(institutionalPageId));
    }

    // original and latest for member by id

    public Page<InstitutionalPageEntity> retrieveAllLatestForMember(UUID userId, Pageable pageable) {
        SearchForMemberInstitutionalPageModel searchFilters = SearchForMemberInstitutionalPageModel.builder().build();
        return retrieveAllLatestForMemberWithFilter(userId, pageable, searchFilters, null);
    }

    public Page<InstitutionalPageEntity> retrieveAllLatestForMemberWithFilter(
            UUID userId,
            Pageable pageable,
            SearchForMemberInstitutionalPageModel searchFilters,
            ModerationStatus moderationStatus
    ) {
        return retrieveAllForMemberWithFilter(
                userId,
                pageable,
                searchFilters,
                moderationStatus != null ? List.of(moderationStatus) : null,
                false,
                true
        );
    }

    public InstitutionalPageEntity retrieveLatestForMemberById(UUID userId, UUID institutionalPageId) {
        SearchForMemberInstitutionalPageModel searchFilters = SearchForMemberInstitutionalPageModel.builder()
                .ids(List.of(institutionalPageId))
                .build();
        return retrieveAllLatestForMemberWithFilter(userId, PageRequest.of(0, 1), searchFilters, null)
                .stream()
                .findFirst()
                .orElseThrow(() -> new InstitutionalPageNotFoundException(institutionalPageId));
    }

    public Page<InstitutionalPageEntity> retrieveAllOriginalForMember(UUID userId, Pageable pageable) {
        SearchForMemberInstitutionalPageModel searchFilters = SearchForMemberInstitutionalPageModel.builder().build();
        return retrieveAllOriginalForMemberWithFilter(userId, pageable, searchFilters, null);
    }

    public Page<InstitutionalPageEntity> retrieveAllOriginalForMemberWithFilter(
            UUID userId,
            Pageable pageable,
            SearchForMemberInstitutionalPageModel searchFilters,
            ModerationStatus moderationStatus
    ) {
        return retrieveAllForMemberWithFilter(
                userId,
                pageable,
                searchFilters,
                moderationStatus != null ? List.of(moderationStatus) : null,
                true,
                false
        );
    }

    public InstitutionalPageEntity retrieveOriginalForMemberById(UUID userId, UUID institutionalPageId) {
        SearchForMemberInstitutionalPageModel searchFilters = SearchForMemberInstitutionalPageModel.builder()
                .ids(List.of(institutionalPageId))
                .build();
        return retrieveAllOriginalForMemberWithFilter(userId, PageRequest.of(0, 1), searchFilters, null)
                .stream()
                .findFirst()
                .orElseThrow(() -> new InstitutionalPageNotFoundException(institutionalPageId));
    }

    public List<InstitutionalPageEntity> retrieveAllOriginalByIds(
            List<UUID> institutionalPageIds
    ) {
        SearchInstitutionalPageModel searchFilters = SearchInstitutionalPageModel.builder()
                .ids(institutionalPageIds)
                .build();
        return retrieveAllOriginalWithFilter(Pageable.unpaged(), searchFilters).getContent();
    }

    public Page<InstitutionalPageEntity> retrieveAllOriginalWithFilter(
            Pageable pageable,
            SearchInstitutionalPageModel searchFilters
    ) {
        return retrieveAllWithFilter(
                pageable,
                searchFilters,
                null,
                true,
                false
        );
    }

    //

    public Page<InstitutionalPageEntity> retrieveAllByViewForMember(
            UUID userId,
            InstitutionalPageView institutionalPageView,
            boolean includePublishedAndNotMember,
            Pageable pageable
    ) {
        SearchForMemberInstitutionalPageModel searchFilters = SearchForMemberInstitutionalPageModel.builder()
                .includePublishedAndNotMember(includePublishedAndNotMember) // to let the user see details of published IPs, even if not member
                .build();
        return retrieveAllByViewForMemberWithFilter(userId, institutionalPageView, pageable, searchFilters);
    }

    public Page<InstitutionalPageEntity> retrieveAllByViewForMemberWithFilter(
            UUID userId,
            InstitutionalPageView institutionalPageView,
            Pageable pageable,
            SearchForMemberInstitutionalPageModel searchFilters
    ) {
        List<ModerationStatus> moderationStatusList = null;
        boolean originalVersionOnly = false;
        boolean latestVersionOnly = false;

        switch (institutionalPageView) {
            case LATEST -> latestVersionOnly = true;
            case ORIGINAL -> originalVersionOnly = true;
            default -> moderationStatusList = List.of(mapper.toModerationStatusEnum(institutionalPageView));
        }
        return retrieveAllForMemberWithFilter(
                userId,
                pageable,
                searchFilters,
                moderationStatusList,
                originalVersionOnly,
                latestVersionOnly
        );
    }

    public InstitutionalPageEntity retrieveByViewForMemberById(
            UUID userId,
            UUID institutionalPageId,
            InstitutionalPageView institutionalPageView,
            boolean includePublishedAndNotMember
    ) {
        SearchForMemberInstitutionalPageModel searchFilters = SearchForMemberInstitutionalPageModel.builder()
                .ids(List.of(institutionalPageId))
                .includePublishedAndNotMember(includePublishedAndNotMember) // to let the user see details of published IPs, even if not member
                .build();
        return retrieveAllByViewForMemberWithFilter(userId, institutionalPageView, PageRequest.of(0, 1), searchFilters)
                .stream()
                .findFirst()
                .orElseThrow(() -> new InstitutionalPageNotFoundException(institutionalPageId));
    }

    // approved for member

    public Page<InstitutionalPageEntity> retrieveAllApprovedForMemberWithFilter(
            UUID userId,
            Pageable pageable,
            SearchForMemberInstitutionalPageModel searchFilters
    ) {
        return retrieveAllForMemberWithFilter(
                userId,
                pageable,
                searchFilters,
                List.of(ModerationStatus.APPROVED),
                true,
                false
        );
    }

    public InstitutionalPageEntity retrieveApprovedForMemberById(UUID userId, UUID institutionalPageId) {
        SearchForMemberInstitutionalPageModel searchFilters = SearchForMemberInstitutionalPageModel.builder()
                .includePublishedAndNotMember(true) // to let the user see details of published IPs, even if not member
                .ids(List.of(institutionalPageId))
                .build();
        return retrieveAllApprovedForMemberWithFilter(userId, PageRequest.of(0, 1), searchFilters)
                .stream()
                .findFirst()
                .orElseThrow(() -> new InstitutionalPageNotFoundException(institutionalPageId));
    }

    public Page<InstitutionalPageEntity> retrieveAllApprovedHierarchyForMemberByRootId(
            UUID userId,
            Pageable pageable,
            UUID rootId
    ) {
        return institutionalPageRepository.findHierarchyForMemberFiltered(
                mapInstitutionalPageSortToColumnName(pageable),
                userId,
                rootId,
                true,
                List.of(ModerationStatus.APPROVED.getLabel()),
                true
        );
    }

    public Page<InstitutionalPageEntity> retrieveAllOriginalInHierarchyForMemberByRootId(
            UUID userId,
            Pageable pageable,
            UUID rootId
    ) {
        return institutionalPageRepository.findHierarchyForMemberFiltered(
                mapInstitutionalPageSortToColumnName(pageable),
                userId,
                rootId,
                false,
                null,
                true
        );
    }

    public Page<InstitutionalPageEntity> retrieveAllOriginalInHierarchyByRootId(
            Pageable pageable,
            UUID rootId
    ) {
        return institutionalPageRepository.findHierarchyFiltered(
                mapInstitutionalPageSortToColumnName(pageable),
                rootId,
                false,
                null,
                true
        );
    }

    // all pending or rejected

    public Page<InstitutionalPageEntity> retrieveAllPendingOrRejected(Pageable pageable) {
        SearchInstitutionalPageModel searchFilters = SearchInstitutionalPageModel.builder().build();
        return retrieveAllPendingOrRejectedWithFilter(pageable, searchFilters, null);
    }

    public Page<InstitutionalPageEntity> retrieveAllPendingOrRejectedWithFilter(
            Pageable pageable,
            SearchInstitutionalPageModel searchFilters,
            String moderationStatus
    ) {
        List<ModerationStatus> moderationStatuses = moderationStatus != null
                ? List.of(ModerationStatus.ofLabel(moderationStatus))
                : List.of(ModerationStatus.PENDING, ModerationStatus.REJECTED);
        return retrieveAllWithFilter(
                pageable,
                searchFilters,
                moderationStatuses,
                false,
                true
        );
    }

    public InstitutionalPageEntity retrievePendingOrRejectedById(UUID institutionalPageId) {
        SearchInstitutionalPageModel searchFilters = SearchInstitutionalPageModel.builder()
                .ids(List.of(institutionalPageId))
                .build();
        return retrieveAllPendingOrRejectedWithFilter(PageRequest.of(0, 1), searchFilters, null)
                .stream()
                .findFirst()
                .orElseThrow(() -> new InstitutionalPageNotFoundException(institutionalPageId));
    }

    // published and approved

    public Page<InstitutionalPageEntity> retrieveAllPublished(Pageable pageable) {
        SearchInstitutionalPageModel searchFilters = SearchInstitutionalPageModel.builder()
                .published(true)
                .build();
        return retrieveAllApprovedWithFilter(pageable, searchFilters);
    }

    public InstitutionalPageEntity retrievePublishedById(UUID institutionalPageId) {
        SearchInstitutionalPageModel searchFilters = SearchInstitutionalPageModel.builder()
                .ids(List.of(institutionalPageId))
                .published(true)
                .build();
        return retrieveAllApprovedWithFilter(PageRequest.of(0, 1), searchFilters)
                .stream()
                .findFirst()
                .orElseThrow(() -> new InstitutionalPageNotFoundException(institutionalPageId));
    }

    public List<InstitutionalPageEntity> retrieveAllApprovedByIds(
            List<UUID> institutionalPageIds
    ) {
        SearchInstitutionalPageModel searchFilters = SearchInstitutionalPageModel.builder()
                .ids(institutionalPageIds)
                .build();
        return retrieveAllApprovedWithFilter(Pageable.unpaged(), searchFilters).getContent();
    }

    public InstitutionalPageEntity retrieveApprovedById(UUID institutionalPageId) {
        SearchInstitutionalPageModel searchFilters = SearchInstitutionalPageModel.builder()
                .ids(List.of(institutionalPageId))
                .build();
        return retrieveAllApprovedWithFilter(PageRequest.of(0, 1), searchFilters)
                .stream()
                .findFirst()
                .orElseThrow(() -> new InstitutionalPageNotFoundException(institutionalPageId));
    }

    public InstitutionalPageEntity retrieveOriginalById(UUID institutionalPageId) {
        return retrieveAllOriginalByIds(List.of(institutionalPageId)).stream().findFirst().orElseThrow(() -> new InstitutionalPageNotFoundException(institutionalPageId));
    }

    public Page<InstitutionalPageEntity> retrieveAllApprovedWithFilter(
            Pageable pageable,
            SearchInstitutionalPageModel searchFilters
    ) {
        return retrieveAllWithFilter(
                pageable,
                searchFilters,
                List.of(ModerationStatus.APPROVED),
                true,
                false
        );
    }

    // navigate relations
    // always retrieve original version

    public UUID retrieveRootInstitutionalPageId(UUID institutionalPageId) {
        InstitutionalPageEntity institutionalPage = retrieveInstitutionalPage(institutionalPageId);
        return getRootInstitutionalPageId(institutionalPage);
    }

    public InstitutionalPageEntity retrieveRootForMemberById(UUID userId, UUID institutionalPageId) {
        InstitutionalPageEntity institutionalPage = retrieveOriginalForMemberById(userId, institutionalPageId);
        return retrieveRootInstitutionalPage(institutionalPage);
    }

    public List<InstitutionalPageEntity> retrieveOriginalRootsInstitutionalPages(List<InstitutionalPageEntity> institutionalPages) {
        List<UUID> institutionalPageIds = new LinkedList<>();
        List<UUID> rootInstitutionalPageIds = new LinkedList<>();
        // map containing institutional pages received as input, which are already root
        Map<UUID, InstitutionalPageEntity> existingRootInstitutionalPages = new HashMap<>();
        for (InstitutionalPageEntity institutionalPage : institutionalPages) {
            UUID institutionalPageId = institutionalPage.getId();
            UUID rootId = InstitutionalPageHelper.getRootInstitutionalPageId(institutionalPage);
            // avoid loading again the same ips
            if (rootId.equals(institutionalPageId) || existingRootInstitutionalPages.containsKey(institutionalPageId)) {
                existingRootInstitutionalPages.put(institutionalPageId, institutionalPage);
            } else {
                institutionalPageIds.add(institutionalPageId);
                rootInstitutionalPageIds.add(rootId);
            }
        }
        if (rootInstitutionalPageIds.isEmpty()) {
            return new LinkedList<>(existingRootInstitutionalPages.values());
        }
        List<InstitutionalPageEntity> foundRoots = new LinkedList<>(retrieveAllOriginalByIds(rootInstitutionalPageIds));
        verifyAllRootFoundOrThrow(institutionalPageIds, foundRoots, rootInstitutionalPageIds);
        // add the root IPs already received as input
        foundRoots.addAll(existingRootInstitutionalPages.values());
        return foundRoots;
    }

    public List<InstitutionalPageEntity> retrieveOriginalParentsInstitutionalPages(List<InstitutionalPageEntity> institutionalPages) {
        List<UUID> institutionalPageIds = new LinkedList<>();
        List<UUID> parentInstitutionalPageIds = new LinkedList<>();
        for (InstitutionalPageEntity institutionalPage : institutionalPages) {
            institutionalPageIds.add(institutionalPage.getId());
            Optional.ofNullable(institutionalPage.getParentInstitutionalPageId())
                    .ifPresent(parentId -> parentInstitutionalPageIds.add(parentId));
        }
        if (parentInstitutionalPageIds.isEmpty()) {
            return new LinkedList<>();
        }
        List<InstitutionalPageEntity> foundParents = retrieveAllOriginalByIds(parentInstitutionalPageIds);
        verifyAllParentFoundOrThrow(institutionalPageIds, foundParents, parentInstitutionalPageIds);
        return foundParents;
    }

    public List<InstitutionalPageEntity> retrieveOriginalChildInstitutionalPages(
            InstitutionalPageEntity institutionalPage
    ) {
        return retrieveOriginalChildInstitutionalPages(List.of(institutionalPage));
    }

    public List<InstitutionalPageEntity> retrieveOriginalChildInstitutionalPages(
            List<InstitutionalPageEntity> institutionalPages
    ) {
        List<UUID> originalInstitutionalPageIds = institutionalPages.stream()
                .map(InstitutionalPageHelper::getOriginalInstitutionalPageId)
                .toList();
        return institutionalPageRepository.findAllByParentInstitutionalPageIdInAndOriginalInstitutionalPageIdIsNull(
                originalInstitutionalPageIds
        );
    }

    public InstitutionalPageEntity retrieveOriginalInstitutionalPage(InstitutionalPageEntity institutionalPage) {
        InstitutionalPageEntity originalInstitutionalPage;
        if (institutionalPage.getOriginalInstitutionalPageId() != null) {
            originalInstitutionalPage = retrieveInstitutionalPageOrThrow(
                    institutionalPage.getOriginalInstitutionalPageId(),
                    () -> new WP2BusinessException(getOriginalIPNotFoundErrorMessage(
                            institutionalPage.getId(),
                            institutionalPage.getOriginalInstitutionalPageId()
                    ))
            );
        } else {
            originalInstitutionalPage = institutionalPage;
        }
        return originalInstitutionalPage;
    }

    public InstitutionalPageEntity retrieveRootInstitutionalPage(InstitutionalPageEntity institutionalPage) {
        InstitutionalPageEntity rootInstitutionalPage;
        if (!CollectionUtils.isEmpty(institutionalPage.getAncestorInstitutionalPageIds())) {
            UUID rootInstitutionalPageId = institutionalPage.getAncestorInstitutionalPageIds().getFirst();
            rootInstitutionalPage = retrieveInstitutionalPageOrThrow(
                    rootInstitutionalPageId,
                    () -> new WP2BusinessException(getRootIPNotFoundErrorMessage(
                            institutionalPage.getId(),
                            rootInstitutionalPageId
                    ))
            );
        } else {
            rootInstitutionalPage = retrieveOriginalInstitutionalPage(institutionalPage);
        }
        return rootInstitutionalPage;
    }

    //

    public static UUID getOriginalInstitutionalPageId(InstitutionalPageEntity institutionalPageEntity) {
        return Optional
                .ofNullable(institutionalPageEntity.getOriginalInstitutionalPageId())
                .orElse(institutionalPageEntity.getId());
    }

    public static UUID getRootInstitutionalPageId(InstitutionalPageEntity institutionalPageEntity) {
        UUID rootInstitutionalPageId;
        if (!CollectionUtils.isEmpty(institutionalPageEntity.getAncestorInstitutionalPageIds())) {
            rootInstitutionalPageId = institutionalPageEntity.getAncestorInstitutionalPageIds().getFirst();
        } else {
            rootInstitutionalPageId = getOriginalInstitutionalPageId(institutionalPageEntity);
        }
        return rootInstitutionalPageId;
    }

    // private

    private Page<InstitutionalPageEntity> retrieveAllWithFilter(
            Pageable pageable,
            SearchInstitutionalPageModel searchFilters,
            List<ModerationStatus> moderationStatuses,
            Boolean originalVersionOnly,
            Boolean latestVersionOnly
    ) {
        boolean applyModerationStatusesFilter = !CollectionUtils.isEmpty(moderationStatuses);
        List<String> moderationStatusLabels = applyModerationStatusesFilter
                ? toStringList(moderationStatuses)
                : List.of("__IGNORED__");
        boolean applyInstitutionalPageIdsFilter = !CollectionUtils.isEmpty(searchFilters.getIds());
        List<UUID> institutionalPageIds = applyInstitutionalPageIdsFilter
                ? searchFilters.getIds()
                : List.of(UUID.fromString("00000000-0000-0000-0000-000000000000"));
        String searchText = Optional.ofNullable(searchFilters.getSearchText())
                .map(String::toLowerCase)
                .orElse(null);

        return institutionalPageRepository.findAllFiltered(
                mapInstitutionalPageSortToColumnName(pageable),
                applyModerationStatusesFilter,
                moderationStatusLabels,
                applyInstitutionalPageIdsFilter,
                institutionalPageIds,
                searchFilters.getPublished(),
                searchFilters.getCategory(),
                searchFilters.getRootLevelOnly() != null && searchFilters.getRootLevelOnly(),
                originalVersionOnly != null && originalVersionOnly,
                latestVersionOnly != null && latestVersionOnly,
                searchText
        );
    }

    private Page<InstitutionalPageEntity> retrieveAllForMemberWithFilter(
            UUID userId,
            Pageable pageable,
            SearchForMemberInstitutionalPageModel searchFilters,
            List<ModerationStatus> moderationStatuses,
            Boolean originalVersionOnly,
            Boolean latestVersionOnly
    ) {
        boolean applyModerationStatusesFilter = !CollectionUtils.isEmpty(moderationStatuses);
        List<String> moderationStatusLabels = applyModerationStatusesFilter
                ? toStringList(moderationStatuses)
                : List.of("__IGNORED__");
        boolean applyInstitutionalPageIdsFilter = !CollectionUtils.isEmpty(searchFilters.getIds());
        List<UUID> institutionalPageIds = applyInstitutionalPageIdsFilter
                ? searchFilters.getIds()
                : List.of(UUID.fromString("00000000-0000-0000-0000-000000000000"));
        String searchText = Optional.ofNullable(searchFilters.getSearchText())
                .map(String::toLowerCase)
                .orElse(null);
        return institutionalPageRepository.findAllForMemberFiltered(
                mapInstitutionalPageSortToColumnName(pageable),
                userId,
                searchFilters.getWpLeaderOnly() != null && searchFilters.getWpLeaderOnly(),
                applyModerationStatusesFilter,
                moderationStatusLabels,
                applyInstitutionalPageIdsFilter,
                institutionalPageIds,
                searchFilters.getIncludePrivateAndMember() == null || searchFilters.getIncludePrivateAndMember(),
                searchFilters.getIncludePublishedAndMember() == null || searchFilters.getIncludePublishedAndMember(),
                searchFilters.getIncludePublishedAndNotMember() != null && searchFilters.getIncludePublishedAndNotMember(),
                searchFilters.getCategory(),
                searchFilters.getRootLevelOnly() != null && searchFilters.getRootLevelOnly(),
                originalVersionOnly != null && originalVersionOnly,
                latestVersionOnly != null && latestVersionOnly,
                searchText,
                ModerationStatus.APPROVED.getLabel()
        );
    }

    //

    private static void verifyAllRootFoundOrThrow(
            List<UUID> institutionalPageIds,
            List<InstitutionalPageEntity> foundRoots,
            List<UUID> requestedRootIds
    ) {
        Set<UUID> foundRootsIds = foundRoots.stream()
                .map(InstitutionalPageEntity::getId)
                .collect(Collectors.toSet());
        Set<UUID> notFoundRootIds = new HashSet<>(requestedRootIds);
        notFoundRootIds.removeAll(foundRootsIds);
        if (!notFoundRootIds.isEmpty()) {
            throw new WP2BusinessException(getRootsErrorMessage(institutionalPageIds, notFoundRootIds));
        }
    }

    private static void verifyAllParentFoundOrThrow(
            List<UUID> institutionalPageIds,
            List<InstitutionalPageEntity> foundParents,
            List<UUID> requestedParentIds
    ) {
        Set<UUID> foundParentsIds = foundParents.stream()
                .map(InstitutionalPageEntity::getId)
                .collect(Collectors.toSet());
        Set<UUID> notFoundParentIds = new HashSet<>(requestedParentIds);
        notFoundParentIds.removeAll(foundParentsIds);
        if (!notFoundParentIds.isEmpty()) {
            throw new WP2BusinessException(getParentsErrorMessage(institutionalPageIds, notFoundParentIds));
        }
    }

    // errors

    private static WP2BusinessException foundNotRelatedIPsError(
            UUID institutionalPageId,
            List<InstitutionalPageEntity> institutionalPages
    ) {
        List<UUID> institutionalPageIds = institutionalPages
                .stream()
                .map(institutionalPageEntity -> institutionalPageEntity.getId())
                .toList();
        return new WP2BusinessException(
                "found more institutional pages for same id '%s' not related to each other, ids: '%s'"
                        .formatted(institutionalPageId, institutionalPageIds));
    }

    private static String getRootIPNotFoundErrorMessage(
            UUID institutionalPageId,
            UUID rootInstitutionalPageId
    ) {
        return "InstitutionalPage with id: '%s', rootInstitutionalPage with id: '%s' not found"
                .formatted(institutionalPageId, rootInstitutionalPageId);
    }

    private static String getRootsErrorMessage(List<UUID> institutionalPageIds, Set<UUID> rootInstitutionalPageIds) {
        return "InstitutionalPage with ids: '%s', some rootInstitutionalPage with ids: '%s' not found"
                .formatted(institutionalPageIds, rootInstitutionalPageIds);
    }

    private static String getParentsErrorMessage(List<UUID> institutionalPageIds, Set<UUID> parentInstitutionalPageIds) {
        return "InstitutionalPage with ids: '%s', some parentInstitutionalPage with ids: '%s' not found"
                .formatted(institutionalPageIds, parentInstitutionalPageIds);
    }

    private static String getOriginalIPNotFoundErrorMessage(
            UUID institutionalPageId,
            UUID originalInstitutionalPageId
    ) {
        return "InstitutionalPage with id: '%s', originalInstitutionalPage with id: '%s' not found"
                .formatted(institutionalPageId, originalInstitutionalPageId);
    }

    //

    private static @NotNull List<String> toStringList(List<ModerationStatus> moderationStatuses) {
        return moderationStatuses.stream()
                .map(moderationStatus -> moderationStatus.getLabel())
                .toList();
    }

    /*
     * Maps the sort using column name which is needed for repository methods that use native queries.
     *
     * @param pageable the pageable to be mapped
     * @return the {@link Pageable} with sort mapped to column name instead of field name
     */
    private static Pageable mapInstitutionalPageSortToColumnName(Pageable pageable) {
        return SpringDataUtils.mapSortToColumnName(pageable, InstitutionalPageEntity.class, true);
    }

}
