package com.finconsgroup.itserr.marketplace.institutionalpage.dm.component;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputSearchForMemberInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputSearchModerationInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputSearchPublishedInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputInstitutionalPageForMembersInHierarchyDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputMembersInHierarchyDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputRelatedInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.InstitutionalPageEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.MemberEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.mapper.InstitutionalPageMapper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.mapper.MemberMapper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.mapper.ParagraphMapper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.model.SearchForMemberInstitutionalPageModel;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.model.SearchInstitutionalPageModel;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.stereotype.Component;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

/**
 * Component responsible for converting {@link InstitutionalPageEntity} instances to their corresponding
 * {@link OutputInstitutionalPageDto} representations, handling association loading and ID normalization.
 *
 * <p>
 * It encapsulates the logic for mapping entities with their requested associations and transforming
 * input filter DTOs to internal search models.
 * </p>
 */
@Component
@RequiredArgsConstructor
@Slf4j
public class DtoBuilder {

    private final InstitutionalPageMapper institutionalPageMapper;
    private final ParagraphMapper paragraphMapper;
    private final MemberMapper memberMapper;
    private final AssociationLoader associationLoader;

    public OutputInstitutionalPageDto mapEntityToDto(UUID userId, InstitutionalPageEntity institutionalPage) {
        return mapEntitiesToDtos(userId, List.of(institutionalPage))
                .stream()
                .findFirst()
                .orElseThrow(() -> new WP2BusinessException("error during mapEntitiesToDtos: not output dto returned"));
    }

    public List<OutputInstitutionalPageDto> mapEntitiesToDtos(
            UUID userId,
            List<InstitutionalPageEntity> institutionalPages) {
        Page<InstitutionalPageEntity> page = new PageImpl<>(institutionalPages);
        return mapEntitiesToDtos(userId, page, Set.of(AssociationLoader.ALL)).getContent();
    }

    public Page<OutputInstitutionalPageDto> mapEntitiesToDtos(
            final UUID userId,
            final Page<InstitutionalPageEntity> institutionalPageEntityPage,
            final Set<String> associationsToLoad
    ) {
        AssociationLoader.LoadedAssociations loadedAssociations = associationLoader.loadAssociations(
                institutionalPageEntityPage,
                associationsToLoad
        );
        return institutionalPageEntityPage
                .map(institutionalPage -> {
                    AssociationLoader.Associations associations = associationLoader.buildAssociations(
                            institutionalPage,
                            associationsToLoad,
                            loadedAssociations
                    );
                    return mapEntityToDtoWithAssociations(userId, institutionalPage, associations);
                });
    }

    // private

    private OutputInstitutionalPageDto mapEntityToDtoWithAssociations(
            final UUID userId,
            final InstitutionalPageEntity institutionalPageEntity,
            final AssociationLoader.Associations associations
    ) {
        // map entity to dto
        OutputInstitutionalPageDto outputInstitutionalPageDto = institutionalPageMapper.toDto(institutionalPageEntity);

        // set root related IP
        outputInstitutionalPageDto.setRootInstitutionalPageName(associations.rootInstitutionalPageName);

        // set parent related IP
        outputInstitutionalPageDto.setParentInstitutionalPage(
                toRelatedInstitutionalPageDto(userId, associations.parentInstitutionalPage)
        );

        // set child related IPs
        outputInstitutionalPageDto.setChildInstitutionalPages(
                Optional.ofNullable(associations.childInstitutionalPages)
                        .orElse(new LinkedList<>())
                        .stream()
                        .map(institutionalPageWithMembers ->
                                toRelatedInstitutionalPageDto(userId, institutionalPageWithMembers)
                        )
                        .toList()
        );

        // set paragraphs
        outputInstitutionalPageDto.setParagraphs(
                Optional.ofNullable(associations.paragraphs)
                        .orElse(new LinkedList<>())
                        .stream()
                        .map(paragraph -> paragraphMapper.toDto(paragraph))
                        .toList()
        );

        // set wpLeaders and members
        outputInstitutionalPageDto.setWpLeads(
                memberMapper.toMemberIds(
                        Optional.ofNullable(associations.wpLeads)
                                .orElse(new LinkedList<>())
                )
        );
        outputInstitutionalPageDto.setMembers(
                memberMapper.toMemberIds(
                        Optional.ofNullable(associations.members)
                                .orElse(new LinkedList<>())
                )
        );

        // set has updated version field
        outputInstitutionalPageDto.setHasUpdatedVersion(institutionalPageEntity.getUpdatedInstitutionalPageId() != null);

        // set id with original institutional page id, if any
        outputInstitutionalPageDto.setId(
                InstitutionalPageHelper.getOriginalInstitutionalPageId(institutionalPageEntity)
        );
        // set root institutional page id, if any
        outputInstitutionalPageDto.setRootInstitutionalPageId(
                InstitutionalPageHelper.getRootInstitutionalPageId(institutionalPageEntity)
        );
        return outputInstitutionalPageDto;
    }


    public SearchForMemberInstitutionalPageModel toFilters(
            final InputSearchForMemberInstitutionalPageDto inputSearchForMemberInstitutionalPageDto
    ) {
        return institutionalPageMapper.toModel(inputSearchForMemberInstitutionalPageDto);
    }

    public SearchInstitutionalPageModel toFilters(
            final InputSearchPublishedInstitutionalPageDto inputSearchPublishedInstitutionalPageDto
    ) {
        SearchInstitutionalPageModel searchInstitutionalPageModel = institutionalPageMapper.toModel(inputSearchPublishedInstitutionalPageDto);
        searchInstitutionalPageModel.setPublished(true);
        return searchInstitutionalPageModel;
    }

    public SearchInstitutionalPageModel toFilters(
            final InputSearchModerationInstitutionalPageDto inputSearchModerationInstitutionalPageDto
    ) {
        return institutionalPageMapper.toModel(inputSearchModerationInstitutionalPageDto);
    }

    public SearchForMemberInstitutionalPageModel toFiltersForMember(
            final InputSearchModerationInstitutionalPageDto inputSearchModerationInstitutionalPageDto
    ) {
        SearchForMemberInstitutionalPageModel searchForMemberInstitutionalPageModel = institutionalPageMapper.toModelForMember(inputSearchModerationInstitutionalPageDto);
        // wpLeaderOnly filter currently not available in inputSearchModerationInstitutionalPageDto
        searchForMemberInstitutionalPageModel.setWpLeaderOnly(false);
        // users shouldn't be able to see pending version of page they are not contributing to, even if IPs are public
        searchForMemberInstitutionalPageModel.setIncludePublishedAndNotMember(false);
        // convert 'published' field to 'includePrivateAndMember' and 'includePublishedAndMember' fields
        if (inputSearchModerationInstitutionalPageDto.getPublished() == null) {
            searchForMemberInstitutionalPageModel.setIncludePrivateAndMember(true);
            searchForMemberInstitutionalPageModel.setIncludePublishedAndMember(true);
        } else if (Boolean.TRUE.equals(inputSearchModerationInstitutionalPageDto.getPublished())) {
            searchForMemberInstitutionalPageModel.setIncludePrivateAndMember(false);
            searchForMemberInstitutionalPageModel.setIncludePublishedAndMember(true);
        } else {
            searchForMemberInstitutionalPageModel.setIncludePrivateAndMember(true);
            searchForMemberInstitutionalPageModel.setIncludePublishedAndMember(false);
        }
        //
        return searchForMemberInstitutionalPageModel;
    }

    public static OutputMembersInHierarchyDto toMembersInHierarchyDto(
            UUID rootInstitutionalPageId,
            Map.Entry<UUID, List<MemberEntity>> entry,
            Map<UUID, OutputInstitutionalPageDto> institutionPageIdToDtoMap
    ) {
        OutputInstitutionalPageDto rootInstitutionalPage = institutionPageIdToDtoMap
                .get(rootInstitutionalPageId);
        List<OutputInstitutionalPageForMembersInHierarchyDto> institutionalPages = entry.getValue()
                .stream()
                .map(member -> getOutputInstitutionalPageForMembersInHierarchyDto(member, institutionPageIdToDtoMap))
                .toList();
        return OutputMembersInHierarchyDto.builder()
                .userId(entry.getKey())
                .rootInstitutionalPage(rootInstitutionalPage)
                .institutionalPages(institutionalPages)
                .build();
    }

    // private

    private OutputRelatedInstitutionalPageDto toRelatedInstitutionalPageDto(
            UUID userId,
            AssociationLoader.InstitutionalPageWithMembers institutionalPageWithMembers
    ) {
        if (institutionalPageWithMembers == null) {
            return null;
        }
        boolean isMember = isMemberOfRelatedInstitutionalPage(userId, institutionalPageWithMembers);
        return institutionalPageMapper.toRelatedInstitutionalPageDto(
                institutionalPageWithMembers.institutionalPage,
                isMember
        );
    }

    private boolean isMemberOfRelatedInstitutionalPage(
            UUID userId,
            AssociationLoader.InstitutionalPageWithMembers institutionalPageWithMembersEntity
    ) {
        List<MemberEntity> memberEntities = new LinkedList<>();
        memberEntities.addAll(
                Optional.ofNullable(institutionalPageWithMembersEntity.getWpLeads())
                        .orElse(new LinkedList<>())
        );
        memberEntities.addAll(
                Optional.ofNullable(institutionalPageWithMembersEntity.getMembers())
                        .orElse(new LinkedList<>())
        );
        Set<UUID> memberIds = new HashSet<>(memberMapper.toMemberIds(memberEntities));
        return memberIds.contains(userId);
    }

    private static OutputInstitutionalPageForMembersInHierarchyDto getOutputInstitutionalPageForMembersInHierarchyDto(
            MemberEntity member,
            Map<UUID, OutputInstitutionalPageDto> institutionPageIdToDtoMap
    ) {
        OutputInstitutionalPageDto outputInstitutionalPageDto = institutionPageIdToDtoMap
                .get(member.getInstitutionalPageId());
        return OutputInstitutionalPageForMembersInHierarchyDto.builder()
                .institutionalPage(outputInstitutionalPageDto)
                .wpLead(member.isWpLead())
                .build();
    }

}
