package com.finconsgroup.itserr.marketplace.institutionalpage.dm.service;

import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputInviteMembersDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputPatchIPInvitationRequestDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputPatchIPJoinRequestDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputPatchMembershipDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputRemoveMembershipDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputSubmitJoinRequestDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputMembersInHierarchyDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputPendingMemberRequestsDto;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;

import java.util.UUID;

/**
 * Service for handling business logic related to Members of InstitutionalPages.
 * <p>
 * This interface provides methods to save join requests and invitations,
 * accept/reject of join-requests/invitations, and promotion to WP Leader.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * OutputPendingMemberRequestsDto pendingRequests = memberService.
 *          findPendingMemberRequestsForInstitutionalPage(userId, institutionalPageId);
 * </pre>
 * </p>
 */
public interface MemberService {

    /**
     * Indicates whether the user has Institutional Page Moderate role.
     *
     * @return a {@link Boolean} with value true if the user has Institutional Page Moderate role
     */
    @NonNull
    Boolean hasIPModerateRole();

    /**
     * Retrieves all pending member join requests and invitations for a specific InstitutionalPage (WP Leader).
     *
     * @param userId              the unique identifier of the authenticated user
     * @param institutionalPageId the id of the InstitutionalPage
     * @return a {@link OutputPendingMemberRequestsDto} representing pending member requests for the institutional page
     */
    @NonNull
    OutputPendingMemberRequestsDto findPendingMemberRequestsForInstitutionalPage(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    );

    /**
     * Retrieves all pending member join requests and invitations related to the authenticated user.
     *
     * @param userId the unique identifier of the authenticated user
     * @return a {@link OutputPendingMemberRequestsDto} representing pending member requests for authenticated user
     */
    @NonNull
    OutputPendingMemberRequestsDto findPendingMemberRequestsForUser(
            @NonNull UUID userId
    );

    /**
     * Invites users to join the specified InstitutionalPage as members (WP Leader).
     *
     * @param userId                the unique identifier of the authenticated user
     * @param institutionalPageId   the id of the InstitutionalPage
     * @param inputInviteMembersDto the input DTO containing invitation information
     * @return a {@link OutputPendingMemberRequestsDto} representing updated pending member requests for the institutional page
     */
    @NonNull
    OutputPendingMemberRequestsDto inviteMembers(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputInviteMembersDto inputInviteMembersDto
    );

    /**
     * Cancels an invitation for a specific user to join an InstitutionalPage (WP Leader).
     *
     * @param userId              the unique identifier of the authenticated user
     * @param institutionalPageId the id of the InstitutionalPage
     * @param invitedUserId       the id of the invited user
     * @return a {@link OutputPendingMemberRequestsDto} representing updated pending member requests for the institutional page
     */
    @NonNull
    OutputPendingMemberRequestsDto cancelInvitation(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull UUID invitedUserId
    );

    /**
     * Accepts/Rejects an invitation to join an InstitutionalPage.
     *
     * @param userId                           the unique identifier of the authenticated user
     * @param institutionalPageId              the id of the InstitutionalPage
     * @param inputPatchIPInvitationRequestDto the input DTO containing accept/reject information
     * @return a {@link OutputPendingMemberRequestsDto} representing updated pending member requests for authenticated user
     */
    @NonNull
    OutputPendingMemberRequestsDto acceptOrRejectInvitation(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputPatchIPInvitationRequestDto inputPatchIPInvitationRequestDto
    );

    /**
     * Updates membership of members to WP Leader/Member for an InstitutionalPage (WP Leader).
     *
     * @param userId                  the unique identifier of the authenticated user
     * @param institutionalPageId     the id of the InstitutionalPage
     * @param inputPatchMembershipDto the input DTO containing  membership updates details
     * @return a {@link OutputInstitutionalPageDto} representing the updated InstitutionalPage
     */
    @NonNull
    OutputInstitutionalPageDto patchMembershipOfUsersForInstitutionalPage(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputPatchMembershipDto inputPatchMembershipDto
    );

    /**
     * Remove membership of users for an InstitutionalPage.
     *
     * @param userId                   the unique identifier of the authenticated user
     * @param institutionalPageId      the id of the InstitutionalPage
     * @param inputRemoveMembershipDto the input DTO containing if of user to remove
     * @return a {@link OutputInstitutionalPageDto} representing the updated InstitutionalPage
     */
    @NonNull
    OutputInstitutionalPageDto removeMembershipOfUsersForInstitutionalPage(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputRemoveMembershipDto inputRemoveMembershipDto
    );

    /**
     * Submits a join request for the authenticated user to join an InstitutionalPage.
     *
     * @param userId                    the unique identifier of the authenticated user
     * @param institutionalPageId       the id of the InstitutionalPage
     * @param inputSubmitJoinRequestDto the input DTO containing join request
     * @return a {@link OutputPendingMemberRequestsDto} representing updated pending member requests for authenticated user
     */
    @NonNull
    OutputPendingMemberRequestsDto submitJoinRequest(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputSubmitJoinRequestDto inputSubmitJoinRequestDto
    );

    /**
     * Cancels a join request previously submitted by the authenticated user.
     *
     * @param userId              the unique identifier of the authenticated user
     * @param institutionalPageId the id of the InstitutionalPage
     * @return a {@link OutputPendingMemberRequestsDto} representing updated pending member requests for authenticated user
     */
    @NonNull
    OutputPendingMemberRequestsDto cancelJoinRequest(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    );

    /**
     * Accepts/Rejects join requests to an institutional page (WP Leader).
     *
     * @param userId                     the unique identifier of the authenticated user
     * @param institutionalPageId        the id of the InstitutionalPage
     * @param inputPatchIPJoinRequestDto the input DTO containing accept/reject join request details
     * @return a {@link OutputPendingMemberRequestsDto} representing updated pending member requests for the institutional page
     */
    @NonNull
    OutputPendingMemberRequestsDto acceptOrRejectJoinRequests(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputPatchIPJoinRequestDto inputPatchIPJoinRequestDto
    );

    /**
     * Retrieves paginated list of all users belonging
     * to the hierarchy of a given root InstitutionalPage (WP Leader).
     *
     * @param userId                  the unique identifier of the authenticated user
     * @param pageable                the pagination information
     * @param rootInstitutionalPageId ID of the root institutional page
     * @return a page of {@link OutputMembersInHierarchyDto} the users with their institutional page associations
     */
    @NonNull
    Page<OutputMembersInHierarchyDto> findAllUsersInHierarchy(
            @NonNull UUID userId,
            @NonNull Pageable pageable,
            @NonNull UUID rootInstitutionalPageId
    );

}
