package com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.impl;

import com.finconsgroup.itserr.marketplace.institutional_page.bs.config.properties.InstitutionalPageBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.InstitutionalPageProducer;
import com.finconsgroup.itserr.messaging.exception.WP2MessagingException;
import com.finconsgroup.itserr.messaging.producer.MessageProducer;
import com.finconsgroup.itserr.messaging.producer.ProducerRegistry;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.Map;

import static com.finconsgroup.itserr.marketplace.institutional_page.bs.util.Constants.SERVICE_FILTER_KEY;
import static com.finconsgroup.itserr.marketplace.institutional_page.bs.util.Constants.SERVICE_FILTER_VALUE;

/**
 * Default implementation of {@link InstitutionalPageProducer} that publishes institutional pages entities.
 */
@Component
@ConditionalOnProperty(name = "messaging.enabled", havingValue = "true")
@Slf4j
public class DefaultInstitutionalPageProducer implements InstitutionalPageProducer {

    private final MessageProducer messageProducer;

    private final InstitutionalPageBsConfigurationProperties.Messaging messagingProperties;

    /**
     * Constructs the DefaultEntityProducer.
     *
     * @param institutionalPageBsConfigurationProperties    application-specific configuration properties
     * @param producerRegistry                              the registry providing message producers
     */
    public DefaultInstitutionalPageProducer(
            InstitutionalPageBsConfigurationProperties institutionalPageBsConfigurationProperties,
            ProducerRegistry producerRegistry
    ) {
        this.messagingProperties = institutionalPageBsConfigurationProperties.getMessaging();
        this.messageProducer = producerRegistry.getMessageProducer(this.messagingProperties.getInstitutionalPageProducerName());
    }

    @Override
    public void publishCreateEvent(OutputInstitutionalPageDto institutionalPage) {
        publish(institutionalPage, messagingProperties.getInstitutionalPageCreatedType());
    }

    @Override
    public void publishUpdateEvent(OutputInstitutionalPageDto institutionalPage) {
        publish(institutionalPage, messagingProperties.getInstitutionalPageUpdatedType());
    }

    @Override
    public void publishDeleteEvent(OutputInstitutionalPageDto institutionalPage) {
        publish(institutionalPage, messagingProperties.getInstitutionalPageDeletedType());
    }

    private void publish(@NonNull OutputInstitutionalPageDto institutionalPage, @NonNull String eventType) {
        try {
            Map<String, OutputInstitutionalPageDto> messageMap = Map.of(institutionalPage.getId().toString(), institutionalPage);
            Map<String, String> filterProperties = Map.of(SERVICE_FILTER_KEY, SERVICE_FILTER_VALUE);
            log.debug("Sending message, messageMap: %s, eventType: %s, source: %s, filterProperties: %s"
                    .formatted(messageMap, eventType, messagingProperties.getSource(), filterProperties));
            messageProducer.publishEvent(messageMap, eventType, messagingProperties.getSource(), filterProperties);
            log.info("Successfully published event message for resource with id: %s".formatted(institutionalPage.getId()));
        } catch (Exception e) {
            String errorMessage = "Error occurred while sending event message: %s".formatted(e.getMessage());
            throw new WP2MessagingException(errorMessage);
        }
    }
}
