package com.finconsgroup.itserr.marketplace.institutional_page.bs.controller;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.api.DiagnosticsApi;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.appender.RollingFileAppender;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RestController;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;

/**
 * REST controller class for handling debugging/analysis related API requests.
 *
 * <p>Implements the {@link DiagnosticsApi} interface.</p>
 */
@Slf4j
@RestController
@RequiredArgsConstructor
public class DiagnosticsController implements DiagnosticsApi {

    // The name of the RollingFile Log4j2 component
    // It is not the name of the log file
    // We use this to dynamically retrieve the name of the log file.
    @Value("${log.log4j2-rolling-file-name}")
    private String log4j2RollingFileName;

    @Override
    public ResponseEntity<Resource> downloadLogs() {
        log.debug("call to DiagnosticsController - downloadLogs");
        Path filePath = Paths.get(getLogFilePathFromLog4j2());
        log.debug("Trying to retrieve log file from: {}", filePath);
        File logFile = filePath.toFile();
        if (!logFile.exists() || !logFile.isFile()) {
            throw new WP2ResourceNotFoundException("Log file not found: %s".formatted(logFile.getAbsolutePath()));
        }
        Resource resource = new FileSystemResource(logFile);
        return ResponseEntity.ok()
                .header(HttpHeaders.CONTENT_DISPOSITION, getContentDispositionHeaderValue(logFile.getName()))
                .contentType(MediaType.APPLICATION_OCTET_STREAM)
                .body(resource);
    }

    // private

    public String getLogFilePathFromLog4j2() {
        LoggerContext context = (LoggerContext) LogManager.getContext(false);
        RollingFileAppender appender = context.getConfiguration().getAppender(log4j2RollingFileName);
        return appender.getFileName();
    }

    private static String getContentDispositionHeaderValue(String fileName) {
        return "attachment; filename=\"%s\"".formatted(fileName);
    }

}