package com.finconsgroup.itserr.marketplace.institutional_page.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2AuthenticationException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.InstitutionalPageDmClient;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.InstitutionalPageIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.InstitutionalPageViewIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.InviteMembersIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.MembersInHierarchyIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.PatchIPInvitationRequestIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.PatchIPJoinRequestIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.PatchMembershipIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.PendingMemberRequestIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.PendingMemberRequestsIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.RemoveMembershipIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.SubmitJoinRequestIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.component.InstitutionalPageEntityProducer;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputInviteMembersDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputPatchIPInvitationRequestDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputPatchIPJoinRequestDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputPatchMembershipDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputRemoveMembershipDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputSubmitJoinRequestDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputMembersInHierarchyDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputPendingMemberRequestsDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.mapper.InstitutionalPageMapper;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.mapper.MemberMapper;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.NotificationMessageFactory;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.NotificationProducer;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.JoinRequestNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.JoinRequestStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MemberInvitationNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MemberInvitationStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MembershipChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MembershipRemovalNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.repository.UserProfileRepository;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.repository.WorkspaceRepository;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.service.MemberService;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import feign.FeignException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultMemberService implements MemberService {
    private static final int INSTITUTIONAL_PAGES_PER_PAGE = 50;

    private final MemberMapper memberMapper;
    private final InstitutionalPageMapper institutionalPageMapper;

    private final InstitutionalPageDmClient institutionalPageDmClient;

    private final InstitutionalPageEntityProducer institutionalPageEntityProducer;
    private final NotificationMessageFactory notificationMessageFactory;
    private final NotificationProducer notificationProducer;
    private final WorkspaceRepository workspaceRepository;
    private final UserProfileRepository userProfileRepository;

    @NotNull
    @Override
    public OutputPendingMemberRequestsDto findPendingMemberRequestsForInstitutionalPage(
            @NotNull UUID institutionalPageId
    ) {
        try {
            PendingMemberRequestsIPDmDto pendingMemberRequestsDmDto =
                    institutionalPageDmClient.findPendingMemberRequestsForInstitutionalPage(institutionalPageId);

            return memberMapper.toOutputPendingMemberRequestsDto(pendingMemberRequestsDmDto);
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @NotNull
    @Override
    public OutputPendingMemberRequestsDto findPendingMemberRequestsForUser() {
        try {
            PendingMemberRequestsIPDmDto pendingMemberRequestsForUserDmDto =
                    institutionalPageDmClient.findPendingMemberRequestsForUser();

            return memberMapper.toOutputPendingMemberRequestsDto(pendingMemberRequestsForUserDmDto);
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @NotNull
    @Override
    public OutputPendingMemberRequestsDto inviteMembers(
            @NotNull UUID institutionalPageId,
            @NotNull InputInviteMembersDto inputInviteMembersDto
    ) {
        try {
            // Check the existence of the user profiles
            List<OutputUserProfileDto> userProfiles = userProfileRepository.getMemberProfilesOrThrow(inputInviteMembersDto.getUserIds());

            InviteMembersIPDmDto inviteMembersIPDmDto = memberMapper.toInviteMembersIPDmDto(inputInviteMembersDto);
            PendingMemberRequestsIPDmDto pendingMemberRequestsIPDmDto =
                    institutionalPageDmClient.inviteMembers(institutionalPageId, inviteMembersIPDmDto);

            // Get the institutional page
            InstitutionalPageIPDmDto institutionalPageIPDmDto = retrieveOriginalInstitutionalPageById(institutionalPageId);

            // Share the institutional page folder with the new users
            workspaceRepository.shareInstitutionalPageFolderToUsers(
                    institutionalPageIPDmDto.getWorkspaceFolderId(),
                    userProfiles
            );

            // Send notification for member invitation
            MessagingEventDto<MemberInvitationNotificationData> notificationMessage = notificationMessageFactory.createMemberInvitationNotification(institutionalPageIPDmDto, inputInviteMembersDto);
            notificationProducer.publishMemberInvitationNotification(notificationMessage);

            return memberMapper.toOutputPendingMemberRequestsDto(pendingMemberRequestsIPDmDto);
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @NotNull
    @Override
    public OutputPendingMemberRequestsDto cancelInvitation(
            @NotNull UUID institutionalPageId,
            @NotNull UUID invitedUserId
    ) {
        try {
            OutputUserProfileDto invitedUserProfile = userProfileRepository.getMemberProfilesOrThrow(List.of(invitedUserId)).getFirst();

            PendingMemberRequestsIPDmDto pendingMemberRequestsIPDmDto =
                    institutionalPageDmClient.cancelInvitation(institutionalPageId, invitedUserId);

            InstitutionalPageIPDmDto institutionalPageIPDmDto = retrieveOriginalInstitutionalPageById(institutionalPageId);

            // Cancel folder sharing with the invited user
            UUID newFolderId = workspaceRepository.unshareInstitutionalPageFolderFromUsers(
                    institutionalPageIPDmDto.getWorkspaceFolderId(),
                    List.of(invitedUserProfile));

            // Update the folder id in the institutional page
            InstitutionalPageIPDmDto newInstitutionalPageIPDmDto = institutionalPageDmClient.updateFolderId(institutionalPageId, newFolderId);

            // Send notification for invitation cancellation
            MessagingEventDto<MemberInvitationNotificationData> notificationMessage = notificationMessageFactory.createMemberInvitationDeletedNotification(newInstitutionalPageIPDmDto, invitedUserId);
            notificationProducer.publishMemberInvitationDeletedNotification(notificationMessage);

            return memberMapper.toOutputPendingMemberRequestsDto(pendingMemberRequestsIPDmDto);
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }


    @NotNull
    @Override
    public OutputPendingMemberRequestsDto acceptOrRejectInvitation(
            @NotNull UUID institutionalPageId,
            @NotNull InputPatchIPInvitationRequestDto inputPatchIPInvitationRequestDto
    ) {
        try {
            // Retrieve the invitation
            Optional<PendingMemberRequestIPDmDto> pendingMemberRequestIPDmDto = institutionalPageDmClient.findPendingMemberRequestsForUser()
                    .getInvitations()
                    .stream()
                    .filter(invitation -> invitation.getInstitutionalPage().getId().equals(institutionalPageId))
                    .findAny();

            // Send the accept/reject request to the DM
            PatchIPInvitationRequestIPDmDto patchIPInvitationRequestIPDmDto =
                    memberMapper.toPatchIPInvitationRequestIPDmDto(inputPatchIPInvitationRequestDto);
            PendingMemberRequestsIPDmDto pendingMemberRequestsDmDto =
                    institutionalPageDmClient.acceptOrRejectInvitation(institutionalPageId, patchIPInvitationRequestIPDmDto);

            if (pendingMemberRequestIPDmDto.isEmpty()) {
                // Invitation accepted/rejected successfully, but the bs couldn't find it initially
                throw new WP2BusinessException("Invitation accepted/rejected successfully, but didn't find invitation fields");
            }
            Boolean wpLeader = pendingMemberRequestIPDmDto.get().getWpLeader();
            InstitutionalPageIPDmDto institutionalPageIPDmDto = pendingMemberRequestIPDmDto.get().getInstitutionalPage();
            List<OutputInstitutionalPageDto> hierarchyInstitutionalPages = List.of();

            if (inputPatchIPInvitationRequestDto.getApproved()) {
                // retrieve the hierarchy
                if (wpLeader != null && wpLeader) {
                    UUID rootInstitutionalPageId = institutionalPageIPDmDto.getRootInstitutionalPageId();
                    hierarchyInstitutionalPages = this.getAllChildInstitutionalPages(rootInstitutionalPageId);
                    // for each institutional page in the hierarchy
                    // if there is an 'Approved' version, e.g., if the original version has status 'Approved'
                    // produce the message with the entity, related to the approved version, not the updated one.
                    hierarchyInstitutionalPages.stream()
                            .filter(InstitutionalPageEntityProducer::isApprovedInstitutionalPage)
                            .forEach(institutionalPageEntityProducer::publishUpdateEvent);
                } else {
                    // only for the specific institutional page
                    // produce the message with the entity, related to the approved version, not the updated one.
                    institutionalPageEntityProducer.publishUpdateEventIfApprovedInstitutionalPageExists(institutionalPageId);
                }

            } else {
                ;
                /*
                TODO
                 due to d4science issues, we currently avoid unsharing as described in
                 https://itserr-wp2.atlassian.net/wiki/spaces/IW/pages/113901693/Technical+Workarounds+for+D4Science
                 we will uncommented the code below when the problem will be solved.
                UUID newFolderId = workspaceRepository.unshareInstitutionalPageFolderFromUsers(
                        institutionalPageIPDmDto.getWorkspaceFolderId(),
                        JwtTokenHolder.getPreferredUsernameOrThrow()
                );
                institutionalPageDmClient.updateFolderId(institutionalPageId, newFolderId);
                */
            }

            // Send notification for invitation status change
            // the notification is created using the institutional page retrieved from invitation
            // avoid sending notification to myself if I've added as wp leader
            MessagingEventDto<MemberInvitationStatusChangeNotificationData> notificationMessage =
                    notificationMessageFactory.createMemberInvitationStatusChangeNotification(institutionalPageIPDmDto, inputPatchIPInvitationRequestDto, wpLeader, hierarchyInstitutionalPages);
            notificationProducer.publishMemberInvitationStatusChangeNotification(notificationMessage);

            return memberMapper.toOutputPendingMemberRequestsDto(pendingMemberRequestsDmDto);
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @NotNull
    @Override
    public OutputInstitutionalPageDto patchMembershipOfUsersForInstitutionalPage(
            @NotNull UUID institutionalPageId,
            @NotNull InputPatchMembershipDto inputPatchMembershipDto
    ) {
        try {
            UUID targetUserId = inputPatchMembershipDto.getUserId();
            // Check the existence of the user profile
            userProfileRepository.getMemberProfilesOrThrow(List.of(targetUserId));

            List<OutputInstitutionalPageDto> institutionalPageHierarchy = List.of();

            // if wp-leader to member retrieve before patch the hierarchy
            if (!inputPatchMembershipDto.getWpLeader()) {
                InstitutionalPageIPDmDto institutionalPageIPDmDto = retrieveOriginalInstitutionalPageById(institutionalPageId);
                if (institutionalPageIPDmDto.getWpLeads().contains(targetUserId)) {
                    institutionalPageHierarchy = this.getAllChildInstitutionalPages(institutionalPageIPDmDto.getRootInstitutionalPageId());
                }
            }

            PatchMembershipIPDmDto patchMembershipIPDmDto = memberMapper.toPatchMembershipIPDmDto(inputPatchMembershipDto);
            InstitutionalPageIPDmDto institutionalPageIPDmDto =
                    institutionalPageDmClient.patchMembershipOfUsersForInstitutionalPage(institutionalPageId, patchMembershipIPDmDto);

            if (inputPatchMembershipDto.getWpLeader()) {
                institutionalPageHierarchy = this.getAllChildInstitutionalPages(institutionalPageIPDmDto.getRootInstitutionalPageId());
            }

            OutputInstitutionalPageDto institutionalPageDto =
                    institutionalPageMapper.toOutputInstitutionalPageDto(institutionalPageIPDmDto, true);

            // Send notification for membership change
            MessagingEventDto<MembershipChangeNotificationData> notificationMessage = notificationMessageFactory.createMembershipChangeNotification(inputPatchMembershipDto, institutionalPageDto, institutionalPageHierarchy);
            notificationProducer.publishMembershipChangeNotification(notificationMessage);

            if (inputPatchMembershipDto.getWpLeader()) {
                // for each institutional page in the hierarchy
                // if there is an 'Approved' version, e.g., if the original version has status 'Approved'
                // produce the message with the entity, related to the approved version, not the updated one.
                institutionalPageHierarchy.stream()
                        .filter(InstitutionalPageEntityProducer::isApprovedInstitutionalPage)
                        .forEach(institutionalPageEntityProducer::publishUpdateEvent);
            } else {
                UUID rootInstitutionalPageId = institutionalPageIPDmDto.getRootInstitutionalPageId();
                // for each institutional page in the hierarchy
                // if there is an 'Approved' version, e.g., if the original version has status 'Approved'
                // produce the message with the entity, related to the approved version, not the updated one.
                institutionalPageHierarchy.stream()
                        .filter(InstitutionalPageEntityProducer::isApprovedInstitutionalPage)
                        .map(outputInstitutionalPageDto -> {
                            // align the institutional page removing the wp leader
                            // the hierarchy list was retrieved before the patch operation.
                            if (outputInstitutionalPageDto.getId().equals(rootInstitutionalPageId)) {
                                OutputUserProfileDto targetUser = outputInstitutionalPageDto.getWpLeads()
                                        .stream()
                                        .filter(userProfile -> userProfile.getId().equals(targetUserId))
                                        .findFirst()
                                        .orElseThrow(() -> new WP2BusinessException("institutional page with id: '%s', user with id: '%s' not found in wpLeads list".formatted(institutionalPageId, targetUserId)));
                                outputInstitutionalPageDto.getWpLeads()
                                        .removeIf(userProfile -> userProfile.getId().equals(targetUserId));
                                outputInstitutionalPageDto.getMembers().add(targetUser);
                            } else {
                                outputInstitutionalPageDto.getWpLeads()
                                        .removeIf(userProfile -> userProfile.getId().equals(targetUserId));
                            }
                            return outputInstitutionalPageDto;
                        })
                        .forEach(institutionalPageEntityProducer::publishUpdateEvent);
            }
            return institutionalPageDto;
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @NotNull
    @Override
    public OutputInstitutionalPageDto removeMembershipOfUsersForInstitutionalPage(
            @NotNull UUID institutionalPageId,
            @NotNull InputRemoveMembershipDto inputRemoveMembershipDto
    ) {
        try {
            UUID targetUserId = inputRemoveMembershipDto.getUserId();
            // Check the existence of the user profile
            userProfileRepository.getMemberProfilesOrThrow(List.of(inputRemoveMembershipDto.getUserId()));

            InstitutionalPageIPDmDto institutionalPageIPDmDto = retrieveOriginalInstitutionalPageById(institutionalPageId);

            boolean wpLeader = institutionalPageIPDmDto
                    .getWpLeads()
                    .contains(inputRemoveMembershipDto.getUserId());

            List <OutputInstitutionalPageDto> hierarchyInstitutionalPages = List.of();
            // if wp-leader, retrieve the hierarchy before removing the user from the institutional page
            if (wpLeader) {
                hierarchyInstitutionalPages = this.getAllChildInstitutionalPages(institutionalPageIPDmDto.getRootInstitutionalPageId());
            }

            RemoveMembershipIPDmDto removeMembershipIPDmDto = memberMapper.toRemoveMembershipIPDmDto(inputRemoveMembershipDto);
            institutionalPageIPDmDto =
                    institutionalPageDmClient.removeMembershipOfUsersForInstitutionalPage(institutionalPageId, removeMembershipIPDmDto);

            OutputInstitutionalPageDto institutionalPageDto =
                    institutionalPageMapper.toOutputInstitutionalPageDto(institutionalPageIPDmDto, true);

            // Send notification for membership removal
            MessagingEventDto<MembershipRemovalNotificationData> notificationMessage =
                    notificationMessageFactory.createMembershipRemovalNotification(inputRemoveMembershipDto, wpLeader, institutionalPageDto, hierarchyInstitutionalPages);
            notificationProducer.publishMembershipRemovalNotification(notificationMessage);

            if (wpLeader) {
                // for each institutional page in the hierarchy
                // if there is an 'Approved' version, e.g., if the original version has status 'Approved'
                // produce the message with the entity, related to the approved version, not the updated one.
                hierarchyInstitutionalPages.stream()
                        .filter(InstitutionalPageEntityProducer::isApprovedInstitutionalPage)
                        .map(outputInstitutionalPageDto -> {
                            // align the institutional page removing the wp leader
                            // the hierarchy list was retrieved before the patch operation.
                            outputInstitutionalPageDto.getWpLeads()
                                    .removeIf(userProfile -> userProfile.getId().equals(targetUserId));
                            return outputInstitutionalPageDto;
                        })
                        .forEach(institutionalPageEntityProducer::publishUpdateEvent);
            } else {
                // only for the specific institutional page
                // produce the message with the entity, related to the approved version, not the updated one.
                institutionalPageEntityProducer.publishUpdateEvent(institutionalPageDto);
            }

            return institutionalPageDto;
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @NotNull
    @Override
    public OutputPendingMemberRequestsDto submitJoinRequest(
            @NotNull UUID institutionalPageId,
            @NotNull InputSubmitJoinRequestDto inputSubmitJoinRequestDto
    ) {
        try {
            SubmitJoinRequestIPDmDto submitJoinRequestIPDmDto = memberMapper.toSubmitJoinRequestIPDmDto(inputSubmitJoinRequestDto);
            PendingMemberRequestsIPDmDto pendingMemberRequestsIPDmDto =
                    institutionalPageDmClient.submitJoinRequest(institutionalPageId, submitJoinRequestIPDmDto);

            // Send notification for join request
            OutputPendingMemberRequestsDto pendingRequests = memberMapper.toOutputPendingMemberRequestsDto(pendingMemberRequestsIPDmDto);
            MessagingEventDto<JoinRequestNotificationData> notificationMessage = notificationMessageFactory.createJoinRequestNotification(institutionalPageId, inputSubmitJoinRequestDto, pendingRequests);
            notificationProducer.publishJoinRequestNotification(notificationMessage);

            return pendingRequests;
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @NotNull
    @Override
    public OutputPendingMemberRequestsDto cancelJoinRequest(
            @NotNull UUID institutionalPageId
    ) {
        try {
            PendingMemberRequestsIPDmDto pendingMemberRequestsIPDmDto =
                    institutionalPageDmClient.cancelJoinRequest(institutionalPageId);

            // Send notification for join request cancellation
            InstitutionalPageIPDmDto institutionalPageIPDmDto = institutionalPageDmClient.getInstitutionalPageById(institutionalPageId, InstitutionalPageViewIPDmDto.APPROVED, true);
            MessagingEventDto<JoinRequestNotificationData> notificationMessage = notificationMessageFactory.createJoinRequestDeletedNotification(institutionalPageIPDmDto);
            notificationProducer.publishJoinRequestDeletedNotification(notificationMessage);

            return memberMapper.toOutputPendingMemberRequestsDto(pendingMemberRequestsIPDmDto);
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @NotNull
    @Override
    public OutputPendingMemberRequestsDto acceptOrRejectJoinRequests(
            @NotNull UUID institutionalPageId,
            @NotNull InputPatchIPJoinRequestDto inputPatchIPJoinRequestDto
    ) {
        try {
            List<OutputUserProfileDto> userProfiles =
                    userProfileRepository.getMemberProfilesOrThrow(inputPatchIPJoinRequestDto.getUserIds());

            PatchIPJoinRequestIPDmDto patchIPJoinRequestIPDmDto = memberMapper.toPatchIPJoinRequestIPDmDto(inputPatchIPJoinRequestDto);
            PendingMemberRequestsIPDmDto pendingMemberRequestsIPDmDto =
                    institutionalPageDmClient.acceptOrRejectJoinRequests(institutionalPageId, patchIPJoinRequestIPDmDto);

            // Send notification for join request status change
            InstitutionalPageIPDmDto institutionalPageIPDmDto = retrieveOriginalInstitutionalPageById(institutionalPageId);

            MessagingEventDto<JoinRequestStatusChangeNotificationData> notificationMessage = notificationMessageFactory.createJoinRequestStatusChangeNotification(institutionalPageIPDmDto, inputPatchIPJoinRequestDto);
            notificationProducer.publishJoinRequestStatusChangeNotification(notificationMessage);

            if (inputPatchIPJoinRequestDto.getApproved()) {
                // produce the message with the entity, related to the approved version, not the updated one.
                institutionalPageEntityProducer.publishUpdateEventIfApprovedInstitutionalPageExists(institutionalPageId);
                // share the institutional page folder with the new user
                workspaceRepository.shareInstitutionalPageFolderToUsers(institutionalPageIPDmDto.getWorkspaceFolderId(), userProfiles);
            }

            return memberMapper.toOutputPendingMemberRequestsDto(pendingMemberRequestsIPDmDto);
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @NotNull
    @Override
    public OutputPageDto<OutputMembersInHierarchyDto> findAllUsersInHierarchy(
            @NotNull UUID rootInstitutionalPageId,
            int pageNumber,
            int pageSize
    ) {
        try {
            OutputPageDto<MembersInHierarchyIPDmDto> membersInHierarchyIPDmDtoPage =
                    institutionalPageDmClient.findAllUsersInHierarchy(rootInstitutionalPageId, pageNumber, pageSize);

            return memberMapper.toOutputMembersInHierarchyDto(membersInHierarchyIPDmDtoPage);
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(rootInstitutionalPageId);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    private List<OutputInstitutionalPageDto> getAllChildInstitutionalPages(UUID institutionalPageId) {
        int pageNo = 0;
        int pageSize = INSTITUTIONAL_PAGES_PER_PAGE;

        List<OutputInstitutionalPageDto> childInstitutionalPages = new ArrayList<>();

        // First fetch
        try {
            OutputPageDto<InstitutionalPageIPDmDto> institutionalPagesPage = institutionalPageDmClient.findInstitutionalPagesHierarchyByRootId(
                    institutionalPageId,
                    Set.of("all"),
                    pageNo,
                    pageSize,
                    "name",
                    SortDirection.ASC
            );
            long fetched = institutionalPagesPage.getContent().size();
            institutionalPagesPage.getContent().stream()
                    .map((institutionalPageIPDmDto) ->
                            institutionalPageMapper.toOutputInstitutionalPageDto(institutionalPageIPDmDto, true))
                    .forEach(childInstitutionalPages::add);

            // Continue to fetch until there are no more elements
            while (fetched == pageSize) {
                pageNo++;
                try {
                    institutionalPagesPage = institutionalPageDmClient.findInstitutionalPagesHierarchyByRootId(
                            institutionalPageId,
                            Set.of("all"),
                            pageNo,
                            pageSize,
                            "name",
                            SortDirection.ASC
                    );
                    fetched = institutionalPagesPage.getContent().size();
                    institutionalPagesPage.getContent().stream()
                            .map((institutionalPageIPDmDto) ->
                                    institutionalPageMapper.toOutputInstitutionalPageDto(institutionalPageIPDmDto, true))
                            .forEach(childInstitutionalPages::add);
                } catch (FeignException.NotFound ex) {
                    log.warn("No child institutional pages found for page number - {}", pageNo, ex);
                    // break out of the loop
                    break;
                }
            }

        } catch (FeignException.NotFound ex) {
            log.warn("No child institutional pages found", ex);
        }

        return childInstitutionalPages;
    }

    private InstitutionalPageIPDmDto retrieveOriginalInstitutionalPageById(UUID institutionalPageId) {
        return institutionalPageDmClient.getInstitutionalPageById(institutionalPageId, InstitutionalPageViewIPDmDto.ORIGINAL, false);
    }

}
