package com.finconsgroup.itserr.marketplace.institutional_page.bs.component;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2FeignClientException;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.InstitutionalPageDmClient;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.InstitutionalPageIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.InstitutionalPageViewIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.mapper.InstitutionalPageMapper;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.InstitutionalPageProducer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;

import java.util.UUID;

/**
 * Publishes event on OpenSearch with the proper version of an Institutional Page.
 */
@Component
@RequiredArgsConstructor
@Slf4j
public class InstitutionalPageEntityProducer {

    private final InstitutionalPageMapper institutionalPageMapper;
    private final InstitutionalPageDmClient institutionalPageDmClient;
    private final InstitutionalPageProducer institutionalPageProducer;

    public void publishCreateEvent(OutputInstitutionalPageDto outputInstitutionalPageDto) {
        institutionalPageProducer.publishCreateEvent(outputInstitutionalPageDto);
    }

    public void publishUpdateEvent(OutputInstitutionalPageDto outputInstitutionalPageDto) {
        if (isApprovedInstitutionalPage(outputInstitutionalPageDto)) {
            institutionalPageProducer.publishUpdateEvent(outputInstitutionalPageDto);
        }
    }

    public void publishDeleteEvent(OutputInstitutionalPageDto outputInstitutionalPageDto) {
        if (isApprovedInstitutionalPage(outputInstitutionalPageDto)) {
            institutionalPageProducer.publishDeleteEvent(outputInstitutionalPageDto);
        }
    }

    public void publishUpdateEventIfApprovedInstitutionalPageExists(final UUID institutionalPageId) {
        InstitutionalPageIPDmDto approvedInstitutionalPageDmDto;
        try {
            approvedInstitutionalPageDmDto = retrieveApprovedInstitutionalPageById(institutionalPageId);
        } catch (WP2FeignClientException exception) {
            if (exception.getHttpStatus() == HttpStatus.NOT_FOUND) {
                // No action needed:
                // this means the institutional page has never been approved,
                // so there's nothing to update in OpenSearch for the approved version.
                return;
            }
            throw exception;
        }

        OutputInstitutionalPageDto approvedInstitutionalPageDto = institutionalPageMapper.toOutputInstitutionalPageDto(approvedInstitutionalPageDmDto, true);
        publishUpdateEvent(approvedInstitutionalPageDto);
    }

    public void publishDeleteApprovedInstitutionalPageEntityIfExists(final UUID institutionalPageId) {
        InstitutionalPageIPDmDto approvedInstitutionalPageDmDto;
        try {
            approvedInstitutionalPageDmDto = retrieveApprovedInstitutionalPageById(institutionalPageId);
        } catch (WP2FeignClientException exception) {
            if (exception.getHttpStatus() == HttpStatus.NOT_FOUND) {
                // No action needed:
                // this means the institutional page has never been approved,
                // so there's nothing to update in OpenSearch for the approved version.
                return;
            }
            throw exception;
        }

        OutputInstitutionalPageDto approvedInstitutionalPageDto = institutionalPageMapper.toOutputInstitutionalPageDto(approvedInstitutionalPageDmDto, true);
        institutionalPageProducer.publishDeleteEvent(approvedInstitutionalPageDto);
    }

    // static

    public static boolean isApprovedInstitutionalPage(OutputInstitutionalPageDto outputInstitutionalPageDto) {
        // TODO should not use InstitutionalPageView.APPROVED.getLabel() to retrieve the constant "Approved"
        // TODO but there is no enum for ModerationStatus on bs currently
        return outputInstitutionalPageDto.getModerationStatus().equals(InstitutionalPageViewIPDmDto.APPROVED.getLabel());
    }

    // private

    private InstitutionalPageIPDmDto retrieveApprovedInstitutionalPageById(final UUID institutionalPageId) {
        return institutionalPageDmClient.getInstitutionalPageById(institutionalPageId, InstitutionalPageViewIPDmDto.APPROVED, false);
    }

}
