package com.finconsgroup.itserr.marketplace.institutional_page.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2AuthenticationException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.InstitutionalPageDmClient;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.InstitutionalPageIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.ModerationIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.component.InstitutionalPageEntityProducer;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputModerationStatusDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputSearchModerationInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.mapper.InstitutionalPageMapper;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.mapper.ModerationMapper;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.NotificationMessageFactory;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.NotificationProducer;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.service.ModerationService;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import feign.FeignException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Set;
import java.util.UUID;

@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultModerationService implements ModerationService {
    private final ModerationMapper moderationMapper;
    private final InstitutionalPageMapper institutionalPageMapper;
    private final InstitutionalPageDmClient institutionalPageDmClient;

    private final InstitutionalPageEntityProducer institutionalPageEntityProducer;
    private final NotificationMessageFactory notificationMessageFactory;
    private final NotificationProducer notificationProducer;

    @Override
    public OutputPageDto<OutputInstitutionalPageDto> getModerationInstitutionalPages(Set<String> associationsToLoad, int pageNumber, int pageSize, String sort, SortDirection direction) {
        try {
            OutputPageDto<InstitutionalPageIPDmDto> pendingInstitutionalPages = institutionalPageDmClient.getModerationInstitutionalPages(
                    associationsToLoad,
                    pageNumber,
                    pageSize,
                    sort,
                    direction
            );

            return institutionalPageMapper.toOutputPageDto(pendingInstitutionalPages);
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @Override
    public OutputInstitutionalPageDto getModerationInstitutionalPageById(UUID institutionalPageId) {
        try {
            InstitutionalPageIPDmDto pendingInstitutionalPage = institutionalPageDmClient.getModerationInstitutionalPageById(institutionalPageId);
            return institutionalPageMapper.toOutputInstitutionalPageDto(pendingInstitutionalPage, true);
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }

    }

    @Override
    public OutputInstitutionalPageDto changeInstitutionalPageModerationStatus(UUID institutionalPageId, InputModerationStatusDto inputModerationStatusDto) {
        try {
            ModerationIPDmDto moderationIPDmDto = institutionalPageDmClient.changeInstitutionalPageModerationStatus(
                    institutionalPageId,
                    moderationMapper.toModerationStatusDmDto(inputModerationStatusDto)
            );

            OutputInstitutionalPageDto institutionalPageDto =
                    institutionalPageMapper.toOutputInstitutionalPageDto(moderationIPDmDto.getInstitutionalPage(), true);

            // Send moderation status change notification
            MessagingEventDto<InstitutionalPageStatusChangeNotificationData> statusChangeNotification = notificationMessageFactory.createInstitutionalPageStatusChangeNotification(inputModerationStatusDto, moderationIPDmDto);
            notificationProducer.publishInstitutionalPageStatusChangeNotification(statusChangeNotification);

            // If approved, produce the message with the entity
            if (inputModerationStatusDto.getApproved()) {
                switch (moderationIPDmDto.getOperationType()) {
                    case CREATE ->
                            institutionalPageEntityProducer.publishCreateEvent(institutionalPageDto);
                    case UPDATE ->
                            institutionalPageEntityProducer.publishUpdateEvent(institutionalPageDto);
                    case DELETE ->
                            institutionalPageEntityProducer.publishDeleteEvent(institutionalPageDto);
                    default -> {
                        log.error("Operation type not supported: {}", moderationIPDmDto.getOperationType());
                        throw new WP2BusinessException("Operation type not supported: " + moderationIPDmDto.getOperationType());
                    }
                }
            }

            return institutionalPageDto;
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @Override
    public OutputPageDto<OutputInstitutionalPageDto> searchModerationInstitutionalPages(
            InputSearchModerationInstitutionalPageDto inputSearchModerationInstitutionalPageDto,
            Set<String> associationsToLoad,
            int pageNumber,
            int pageSize,
            String sort,
            SortDirection direction) {
        try {
            OutputPageDto<InstitutionalPageIPDmDto> clientResponse = institutionalPageDmClient.searchModerationInstitutionalPages(
                    inputSearchModerationInstitutionalPageDto,
                    associationsToLoad,
                    pageNumber,
                    pageSize,
                    sort,
                    direction
            );

            return clientResponse.map(institutionalPageIPDmDto -> 
                    institutionalPageMapper.toOutputInstitutionalPageDto(institutionalPageIPDmDto, true));
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

}
