package com.finconsgroup.itserr.marketplace.institutional_page.bs.api;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputCreateInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputSearchForMemberInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputUpdateInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InstitutionalPageView;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputRequestUpdateDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputWorkspaceItemDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.validation.ValidAssociationToLoad;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Encoding;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RequestPart;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.multipart.MultipartFile;

import java.util.List;
import java.util.Set;
import java.util.UUID;

/**
 * CRUD operations for institutional pages.
 */
@Tag(name = "Institutional pages", description = "Operations related to the institutional pages")
@SecurityRequirement(name = "BearerAuth")
public interface InstitutionalPageApi {
    /**
     * Get all institutional pages for which the user is a contributor
     *
     * @param view the selection strategy to apply (default: {@code LATEST})
     * @param includePublishedAndNotMember The flag to indicate if published institutional pages for which the user is NOT a member should be returned, default value is false.
     * @param associationsToLoad comma separated list of the associations to be returned (default is "all").
     * @param pageNumber the page number
     * @param pageSize the size of the page
     * @param sort the attribute to sort by
     * @param direction the sorting direction (ASC or DESC)
     * @return the dto of the institutional page
     */
    @Operation(summary = "Get all institutional pages")
    @ApiResponses(value = @ApiResponse(responseCode = "200", description = "OK"))
    @GetMapping(path = "/institutional-page/institutional-pages", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputPageDto<OutputInstitutionalPageDto>> getAllInstitutionalPages(
            @RequestParam(name = "view", required = false, defaultValue = "LATEST") InstitutionalPageView view,
            @RequestParam(name = "includePublishedAndNotMember", defaultValue = "false", required = false) boolean includePublishedAndNotMember,
            @Schema(allowableValues = {"all", "none", "parentInstitutionalPage", "wpLeads", "members", "childInstitutionalPages", "paragraphs"})
            @RequestParam(name = "associationsToLoad", defaultValue = "all", required = false) Set<@ValidAssociationToLoad String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name ="sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) SortDirection direction
    );

    /**
     * Get institutional page by id, only if the user is a member.
     * @param institutionalPageId the id of the institutional page to retrieve
     * @param view the selection strategy to apply (default: {@code LATEST})
     * @param includePublishedAndNotMember The flag to indicate if published institutional pages for which the user is NOT a member should be returned, default value is false.
     * @return the dto of the institutional page
     */
    @Operation(summary = "Get institutional page by id, only if the user is a member")
    @ApiResponses(value = @ApiResponse(responseCode = "200", description = "OK"))
    @GetMapping(path = "/institutional-page/institutional-pages/{institutionalPageId}", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputInstitutionalPageDto> getInstitutionalPageById(
            @PathVariable UUID institutionalPageId,
            @RequestParam(name = "view", required = false, defaultValue = "LATEST") InstitutionalPageView view,
            @RequestParam(name = "includePublishedAndNotMember", defaultValue = "false", required = false) boolean includePublishedAndNotMember
    );

    /**
     * Create an institutional page
     *
     * @param institutionalPageDto the body with the institutional page information
     * @param imageFile optional image file to upload
     * @return the created institutional page
     */
    @Operation(
            summary = "Create an institutional page",
            requestBody = @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true,
                    content = @Content(
                            mediaType = MediaType.MULTIPART_FORM_DATA_VALUE,
                            encoding = {
                                    @Encoding(name = "institutionalPageDto", contentType = MediaType.APPLICATION_JSON_VALUE),
                                    @Encoding(name = "imageFile", contentType = "image/*")
                            }
                    )
            )
    )
    @ApiResponses(@ApiResponse(responseCode = "201", description = "Created"))
    @PostMapping(
            value = "/institutional-page/institutional-pages",
            consumes = MediaType.MULTIPART_FORM_DATA_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    ResponseEntity<OutputInstitutionalPageDto> createInstitutionalPage(
            @Valid @RequestPart("institutionalPageDto") InputCreateInstitutionalPageDto institutionalPageDto,
            @RequestPart(value = "imageFile", required = false) MultipartFile imageFile
    );

    /**
     * Update an institutional page by id, only if the user is a member.
     *
     * @param institutionalPageId  the id of the institutional page to update
     * @param institutionalPageDto the body with the institutional page information
     * @param imageFile optional image file to upload
     * @return the updated institutional page
     */
    @Operation(
            summary = "Update an institutional page by id, only if the user is a member",
            requestBody = @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true,
                    content = @Content(
                            mediaType = MediaType.MULTIPART_FORM_DATA_VALUE,
                            encoding = {
                                    @Encoding(name = "institutionalPageDto", contentType = MediaType.APPLICATION_JSON_VALUE),
                                    @Encoding(name = "imageFile", contentType = "image/*")
                            }
                    )
            )
    )
    @ApiResponses(value = @ApiResponse(responseCode = "200", description = "OK"))
    @PutMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}",
            consumes = MediaType.MULTIPART_FORM_DATA_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    ResponseEntity<OutputInstitutionalPageDto> updateInstitutionalPage(
            @PathVariable UUID institutionalPageId,
            @Valid @RequestPart("institutionalPageDto") InputUpdateInstitutionalPageDto institutionalPageDto,
            @RequestPart(value = "imageFile", required = false) MultipartFile imageFile
    );

    /**
     * Deletes an institutional page by id, only if the user is a member.
     *
     * @param institutionalPageId the id of the institutional page to delete
     */
    @Operation(summary = "Delete an institutional page by id, only if the user is a member")
    @ApiResponses(value = @ApiResponse(responseCode = "204", description = "No content"))
    @DeleteMapping("/institutional-page/institutional-pages/{institutionalPageId}")
    ResponseEntity<Void> deleteInstitutionalPage(@PathVariable UUID institutionalPageId);

    /**
     * Retrieves a paginated list of all InstitutionalPages the user is contributing to, matching the search criteria.
     *
     * @param view                            the selection strategy to apply (default: {@code LATEST})
     * @param inputSearchForMemberInstitutionalPageDto the dto containing the filters to be applied
     * @param associationsToLoad              comma separated list of the associations to be returned (default is "all").
     * @param pageNumber                      the page number to retrieve (default is 0)
     * @param pageSize                        the number of InstitutionalPages per page (default is 10)
     * @param sort                            the field to sort by (default is "name")
     * @param direction                       the direction of sorting (default is ascending)
     * @return a page of {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Find all InstitutionalPages the user is contributing to,matching the search criteria",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @PostMapping(value = "/institutional-page/institutional-pages/search", consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputPageDto<OutputInstitutionalPageDto>> search(
            @RequestParam(name = "view", required = false, defaultValue = "LATEST") InstitutionalPageView view,
            @Valid @RequestBody InputSearchForMemberInstitutionalPageDto inputSearchForMemberInstitutionalPageDto,
            @Schema(allowableValues = {"all", "none", "parentInstitutionalPage", "wpLeads", "members", "childInstitutionalPages", "paragraphs"})
            @RequestParam(name = "associationsToLoad", defaultValue = "all", required = false) Set<@ValidAssociationToLoad String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) SortDirection direction
    );

    /**
     * Retrieves a paginated list of all InstitutionalPages the user is contributing to,
     * corresponding to the hierarchy associated with the rootInstitutionalPageId.
     *
     * @param associationsToLoad comma separated list of the associations to be returned (default is "all").
     * @param pageNumber         the page number to retrieve (default is 0)
     * @param pageSize           the number of InstitutionalPages per page (default is 10)
     * @param sort               the field to sort by (default is "name")
     * @param direction          the direction of sorting (default is ascending)
     * @return a page of {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves a paginated list of all InstitutionalPages the user is contributing to, " +
                    "corresponding to the hierarchy associated to the rootInstitutionalPageId",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/hierarchy",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    ResponseEntity<OutputPageDto<OutputInstitutionalPageDto>> findInstitutionalPagesHierarchyByRootId(
            @PathVariable("institutionalPageId") UUID rootInstitutionalPageId,
            @Schema(allowableValues = {"all", "none", "parentInstitutionalPage", "wpLeads", "members", "childInstitutionalPages", "paragraphs"})
            @RequestParam(name = "associationsToLoad", defaultValue = "all", required = false) Set<@ValidAssociationToLoad String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) SortDirection direction
    );

    /**
     * Retrieve folder content related to institutional page
     *
     * @param institutionalPageId the id of the institutional page
     * @param pageNumber          the page number
     * @param pageSize            the size of the page
     * @return the list of items contained in the folder
     */
    @Operation(summary = "Retrieve folder content related to institutional page")
    @ApiResponses(value = @ApiResponse(responseCode = "200", description = "OK"))
    @GetMapping(path = "/institutional-page/institutional-pages/{institutionalPageId}/folder-content", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<List<OutputWorkspaceItemDto>> retrieveFolderContent(
            @PathVariable UUID institutionalPageId,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize
    );

    /**
     * Request moderation for a specific InstitutionalPage by id, only if the user is a member.
     *
     * @param institutionalPageId the id of the InstitutionalPage to request moderation for
     * @return the updated {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Request moderation for a specific InstitutionalPage by id, only if the user is a member",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @PutMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/request-moderation",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputInstitutionalPageDto requestModerationInstitutionalPage(
            @PathVariable("institutionalPageId") UUID institutionalPageId
    );

    /**
     * Request publication for a specific InstitutionalPage by id, only if the user is a member.
     *
     * @param institutionalPageId the id of the InstitutionalPage to request publication for
     * @return the updated {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Request publication for a specific InstitutionalPage by id, only if the user is a member",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @PutMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/request-publication",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputInstitutionalPageDto requestPublicationInstitutionalPage(
            @PathVariable("institutionalPageId") UUID institutionalPageId
    );

    /**
     * Request update for a specific InstitutionalPage by id, only if the user is a member
     *
     * @param institutionalPageId the id of the InstitutionalPage to update
     * @return the updated {@link OutputRequestUpdateDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Request update for a specific InstitutionalPage by id, only if the user is a member",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @PutMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/request-update",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    ResponseEntity<OutputRequestUpdateDto> requestUpdateInstitutionalPageById(
            @PathVariable("institutionalPageId") UUID institutionalPageId
    );

    /**
     * Release lock-for-update of a specific InstitutionalPage by id, only if the user is a member.
     *
     * @param institutionalPageId the id of the InstitutionalPage to update
     */
    @Operation(
            summary = "Release lock-for-update of a specific InstitutionalPage by id, only if the user is a member",
            responses = {
                    @ApiResponse(responseCode = "204", description = "No Content"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @PutMapping(value = "/institutional-page/institutional-pages/{institutionalPageId}/cancel-update")
    @ResponseStatus(HttpStatus.NO_CONTENT)
    ResponseEntity<Void> cancelUpdateInstitutionalPageById(
            @PathVariable("institutionalPageId") UUID institutionalPageId
    );

}
