package com.finconsgroup.itserr.marketplace.institutional_page.bs.dto;

import io.swagger.v3.oas.annotations.media.Schema;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotBlank;
import jakarta.validation.constraints.NotNull;
import jakarta.validation.constraints.Pattern;
import jakarta.validation.constraints.Size;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.institutional_page.bs.util.Constants.MAX_LONG_TEXT_LENGTH;
import static com.finconsgroup.itserr.marketplace.institutional_page.bs.util.Constants.MAX_MEDIUM_TEXT_LENGTH;
import static com.finconsgroup.itserr.marketplace.institutional_page.bs.util.Constants.MAX_SUPPORTED_OS_LENGTH;
import static com.finconsgroup.itserr.marketplace.institutional_page.bs.util.Constants.VALIDATION_ERROR_FIELD_IP_CATEGORY_ALLOWED_VALUES;
import static com.finconsgroup.itserr.marketplace.institutional_page.bs.util.Constants.VALIDATION_ERROR_FIELD_LENGTH;
import static com.finconsgroup.itserr.marketplace.institutional_page.bs.util.Constants.VALIDATION_ERROR_REQUIRED_FIELD;

@Schema(name = "InputUpdateInstitutionalPageDto", description = "Update DTO of an institutionalPage")
@Data
@Builder
@AllArgsConstructor
@NoArgsConstructor
public class InputUpdateInstitutionalPageDto {
    @Schema(description = "The name of the institutional page", example = "Criterion")
    @NotNull(message = VALIDATION_ERROR_REQUIRED_FIELD)
    @Size(max = MAX_MEDIUM_TEXT_LENGTH, message = VALIDATION_ERROR_FIELD_LENGTH)
    private String name;

    @Schema(description = "ID of the maintainer of the institutional page", example = "33333333-aaaa-1111-0000-000000000000")
    private UUID maintainer;

    @Schema(
            description = "The category of the institutional page",
            allowableValues = {"tool", "project"},
            example = "project"
    )
    @Pattern(regexp = "tool|project", message = VALIDATION_ERROR_FIELD_IP_CATEGORY_ALLOWED_VALUES)
    @NotNull(message = VALIDATION_ERROR_REQUIRED_FIELD)
    private String category;

    @Schema(
            description = "The abstract of the institutional page",
            example = "WP3, the Toolkit for Religious Studies (T-ReS), aims to improve the research experience."
    )
    @NotBlank(message = VALIDATION_ERROR_REQUIRED_FIELD)
    @Size(max = MAX_LONG_TEXT_LENGTH, message = VALIDATION_ERROR_FIELD_LENGTH)
    private String abstractContent;

    @Schema(description = "Copyright info for the institutional page")
    @NotNull(message = VALIDATION_ERROR_REQUIRED_FIELD)
    @Valid
    private InputCopyrightDto copyright;

    @Schema(description = "The paragraphs related to the institutional page")
    private List<@Valid InputParagraphDto> paragraphs;

    @Schema(description = "List of tags", example = "[\"aramaic\",\"english\",\"italian\"]")
    private List<
            @NotBlank(message = VALIDATION_ERROR_REQUIRED_FIELD) @Size(max = 100, message = VALIDATION_ERROR_FIELD_LENGTH) String
            > tags;

    @Schema(description = "List of languages", example = "[\"English\",\"Italian\",\"German\"]")
    private List<
            @NotBlank(message = VALIDATION_ERROR_REQUIRED_FIELD) @Size(max = MAX_MEDIUM_TEXT_LENGTH, message = VALIDATION_ERROR_FIELD_LENGTH) String
            > languages;

    @Schema(description = "The image URL", example = "https://picsum.photos/200")
    @Size(max = MAX_MEDIUM_TEXT_LENGTH, message = VALIDATION_ERROR_FIELD_LENGTH)
    private String imageUrl;

    @Schema(description = "Image metadata for institutional page")
    @Valid
    private InputImageMetadataDto imageMetadata;

    @Schema(description = "The user manual URL", example = "https://www.usermanual.com")
    @Size(max = MAX_MEDIUM_TEXT_LENGTH, message = VALIDATION_ERROR_FIELD_LENGTH)
    private String userManualUrl;

    @Schema(description = "The FAQs page URL", example = "https://www.faqs.com")
    @Size(max = MAX_MEDIUM_TEXT_LENGTH, message = VALIDATION_ERROR_FIELD_LENGTH)
    private String faqsUrl;

    @Schema(description = "The application version", example = "v1.3.0")
    @Size(max = MAX_MEDIUM_TEXT_LENGTH, message = VALIDATION_ERROR_FIELD_LENGTH)
    private String applicationVersion;

    @Schema(description = "The application type", example = "web application")
    @Size(max = MAX_MEDIUM_TEXT_LENGTH, message = VALIDATION_ERROR_FIELD_LENGTH)
    private String applicationType;

    @Schema(description = "The license name", example = "MIT License")
    @Size(max = MAX_MEDIUM_TEXT_LENGTH, message = VALIDATION_ERROR_FIELD_LENGTH)
    private String license;

    @Schema(description = "The supported operating systems for the institutional page", example = "Windows 10+, macOS 12+, Ubuntu 20.04+")
    @Size(max = MAX_SUPPORTED_OS_LENGTH, message = VALIDATION_ERROR_FIELD_LENGTH)
    private String supportedOs;

    @Schema(description = "Input DTO of repository links for the institutional page")
    @Valid
    private InputRepositoriesDto repositories;

    @Schema(
            description = "Additional arbitrary fields in key-value format",
            example = "{\"developer\": \"John Doe\", \"releaseNotes\": \"Initial beta release\"}"
    )
    private LinkedHashMap<
            @NotBlank @Size(max = MAX_MEDIUM_TEXT_LENGTH, message = VALIDATION_ERROR_FIELD_LENGTH) String,
            @NotBlank @Size(max = MAX_MEDIUM_TEXT_LENGTH, message = VALIDATION_ERROR_FIELD_LENGTH) String
            > additionalFields;

    @Schema(description = "External links")
    private LinkedHashMap<String, String> externalLinks;

    @Schema(description = "Buttons related to the institutional page")
    private List<@Valid InputButtonDto> buttons;
}