package com.finconsgroup.itserr.marketplace.institutional_page.bs.client.bs;

import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.bs.dto.CreateDocumentWorkspaceBsDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.bs.dto.CreateFolderWorkspaceBsDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.bs.dto.DocumentPublicLinkWorkspaceBsDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.bs.dto.DocumentWorkspaceBsDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.bs.dto.OutputItemDtoWorkspaceBsDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.bs.dto.ShareFolderWorkspaceBsDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.bs.dto.WorkspaceIdWorkspaceBsDto;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import java.util.List;
import java.util.UUID;

public interface WorkspaceBsClient {
    /**
     * Retrieves the workspace id of the current user.
     *
     * @return an instance of {@link WorkspaceIdWorkspaceBsDto} containing the workspace id
     * @throws feign.FeignException.Unauthorized if the user is not authenticated
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     */
    @GetMapping(path = "/workspace", produces = MediaType.APPLICATION_JSON_VALUE)
    WorkspaceIdWorkspaceBsDto getWorkspace();

    /**
     * Create a new folder
     *
     * @param parentFolderId the folder containing the folder to be created
     * @param createFolderDto the dto representing the details of the folder to be created
     * @return the id of the created folder
     * @throws feign.FeignException.Unauthorized if the user is not authenticated
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     */
    @PostMapping(path = "/workspace/folders/{parentFolderId}/folders")
    String createFolder(@PathVariable UUID parentFolderId, @RequestBody CreateFolderWorkspaceBsDto createFolderDto);

    /**
     * Share a folder with the given users
     *
     * @param folderId the id of the folders to be shared
     * @param shareFolderDto the dto containing the users to which share the folder
     * @throws feign.FeignException.Unauthorized if the user is not authenticated
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     */
    @PostMapping(path = "/workspace/folders/{folderId}/share-users")
    void shareFolder(@PathVariable UUID folderId, @RequestBody ShareFolderWorkspaceBsDto shareFolderDto);

    /**
     * Unshare the folder from the given users
     *
     * @param folderId the id of the folders to be unshared
     * @param usernames the preferredUsername of the users to which unshare the folder
     * @return the new id of the unshared folder
     * @throws feign.FeignException.Unauthorized if the user is not authenticated
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     */
    @DeleteMapping(path = "/workspace/folders/{folderId}/share-users")
    String unshareFolder(@PathVariable UUID folderId, @RequestBody List<String> usernames);

    /**
     * Creates an empty document within a specified folder in the workspace.
     *
     * @param folderId the ID of the folder where the document will be created
     * @param documentDto an object containing the metadata for the document, such as name and description
     * @return an instance of {@code DocumentWorkspaceBsDto} containing the details of the created document
     */
    @PostMapping(path = "/workspace/folders/{folderId}/documents", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    DocumentWorkspaceBsDto createEmptyDocument(
            @PathVariable(name = "folderId") UUID folderId,
            @RequestBody CreateDocumentWorkspaceBsDto documentDto);

    /**
     * Retrieves the public link for a specific document in the workspace.
     *
     * @param documentId the unique identifier of the document for which the public link is requested
     * @return an instance of {@code DocumentPublicLinkWorkspaceBsDto} containing the public link details of the document
     */
    @GetMapping(path = "/workspace/documents/{documentId}/publiclink", produces = MediaType.APPLICATION_JSON_VALUE)
    DocumentPublicLinkWorkspaceBsDto getDocumentPublicLink(
            @PathVariable(name = "documentId") UUID documentId);

    /**
     * Retrieve the user's workspace children (documents and folder), including hidden items (internal use only).
     *
     * @param folderId The ID of the folder containing the documents.
     * @throws feign.FeignException.Unauthorized if the user is not authenticated
     * @throws feign.FeignException.InternalServerError if there is an internal server error
     * @return A list of {@link OutputItemDtoWorkspaceBsDto} representing the documents or subfolders.
     */
    @GetMapping(path = "workspace/folders/{folderId}/children", produces = MediaType.APPLICATION_JSON_VALUE)
    List<OutputItemDtoWorkspaceBsDto> getWorkspaceChildrenByFolderId(
            @PathVariable String folderId,
            @RequestParam(name = "pageNo") Integer pageNo,
            @RequestParam(name = "pageSize") Integer pageSize
    );

}
