package com.finconsgroup.itserr.marketplace.institutional_page.bs.api;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputSearchPublishedInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.validation.ValidAssociationToLoad;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import java.util.Set;
import java.util.UUID;

/**
 * API for getting published institutional pages.
 */
@Tag(name = "Published institutional pages", description = "Operations related to the published institutional pages")
public interface PublishedInstitutionalPageApi {
    /**
     * Get all published institutional pages
     *
     * @param associationsToLoad comma separated list of the associations to be returned (default is "all").
     * @param pageNumber the page number
     * @param pageSize the size of the page
     * @param sort the attribute to sort by
     * @param direction the sorting direction (ASC or DESC)
     * @param withPublications flag indicating whether to include the publication information in the response
     * @return the dto of the institutional page
     */
    @Operation(summary = "Get all institutional pages published")
    @ApiResponses(value = @ApiResponse(responseCode = "200", description = "OK"))
    @GetMapping(path = "/institutional-page/institutional-pages/published", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputPageDto<OutputInstitutionalPageDto>> getAllPublishedInstitutionalPages(
            @RequestParam(name = "associationsToLoad", defaultValue = "all", required = false) Set<@ValidAssociationToLoad String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name ="sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) SortDirection direction,
            @RequestParam(name = "withPublications", defaultValue = "true", required = false) boolean withPublications
    );

    /**
     * Get institutional pages published by id
     * @param institutionalPageId the id of the institutional page to retrieve
     * @param withPublications flag indicating whether to include the publication information in the response
     * @return the dto of the institutional page
     */
    @Operation(summary = "Get institutional page published by id")
    @ApiResponses(value = @ApiResponse(responseCode = "200", description = "OK"))
    @GetMapping(path = "/institutional-page/institutional-pages/published/{institutionalPageId}", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputInstitutionalPageDto> getPublishedInstitutionalPageById(
            @PathVariable UUID institutionalPageId,
            @RequestParam(name = "withPublications", defaultValue = "true", required = false) boolean withPublications
    );

    /**
     * Retrieves a paginated list of all InstitutionalPages published matching the search criteria.
     *
     * @param inputSearchInstitutionalPageDto the dto containing the filters to be applied
     * @param associationsToLoad              comma separated list of the associations to be returned (default is "all").
     * @param pageNumber                      the page number to retrieve (default is 0)
     * @param pageSize                        the number of InstitutionalPages per page (default is 10)
     * @param sort                            the field to sort by (default is "name")
     * @param direction                       the direction of sorting (default is ascending)
     * @param withPublications flag indicating whether to include the publication information in the response
     * @return a page of {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Find all InstitutionalPages published matching the search criteria",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @PostMapping(value = "/institutional-page/institutional-pages/published/search", consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputPageDto<OutputInstitutionalPageDto>> search(
            @Valid @RequestBody InputSearchPublishedInstitutionalPageDto inputSearchInstitutionalPageDto,
            @RequestParam(name = "associationsToLoad", defaultValue = "all", required = false) Set<@ValidAssociationToLoad String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) SortDirection direction,
            @RequestParam(name = "withPublications", defaultValue = "true", required = false) boolean withPublications
    );
}
