package com.finconsgroup.itserr.marketplace.institutional_page.bs.api;

import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputParagraphDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputInstitutionalPageDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import java.util.UUID;

/**
 * CRUD operations for paragraphs of institutional pages.
 */
@Tag(name = "Paragraphs", description = "Operations related to the paragraphs of the institutional pages")
@SecurityRequirement(name = "BearerAuth")
public interface ParagraphApi {

    /**
     * Create a paragraph for an institutional page
     *
     * @param institutionalPageId the id of the institutional page
     * @param paragraphDto the paragraph to create
     * @param position the position where to place the paraph, at the end if not provided
     * @return the institutional page with the created paragraph
     */
    @Operation(summary = "Create a paragraph for an institutional page")
    @ApiResponses(value = @ApiResponse(responseCode = "201", description = "Created"))
    @PostMapping("/institutional-page/institutional-pages/{institutionalPageId}/paragraphs")
    ResponseEntity<OutputInstitutionalPageDto> createParagraph(
            @PathVariable UUID institutionalPageId,
            @Valid @RequestBody InputParagraphDto paragraphDto,
            @RequestParam(required = false) Integer position
    );

    /**
     * Update a paragraph of an institutional page
     *
     * @param institutionalPageId the id of the institutional page
     * @param paragraphId the id of the paragraph to update
     * @param paragraphDto the paragraph to update
     * @return the institutional page with the updated paragraph
     */
    @Operation(summary = "Update a paragraph of an institutional page")
    @ApiResponses(value = @ApiResponse(responseCode = "200", description = "OK"))
    @PutMapping("/institutional-page/institutional-pages/{institutionalPageId}/paragraphs/{paragraphId}")
    ResponseEntity<OutputInstitutionalPageDto> updateParagraph(
            @PathVariable UUID institutionalPageId,
            @PathVariable UUID paragraphId,
            @Valid @RequestBody InputParagraphDto paragraphDto
    );

    /**
     * Delete a paragraph of an institutional page
     *
     * @param institutionalPageId the id of the institutional page
     * @param paragraphId the id of the paragraph to delete
     * @return the institutional page with the deleted paragraph
     */
    @Operation(summary = "Delete a paragraph of an institutional page")
    @ApiResponses(value = @ApiResponse(responseCode = "200", description = "OK"))
    @DeleteMapping("/institutional-page/institutional-pages/{institutionalPageId}/paragraphs/{paragraphId}")
    ResponseEntity<OutputInstitutionalPageDto> deleteParagraph(
            @PathVariable UUID institutionalPageId,
            @PathVariable UUID paragraphId
    );
}
