package com.finconsgroup.itserr.marketplace.favourite.user.dm.service;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2DuplicateResourceException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.dto.InputCreateFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.dto.InputPatchFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.dto.OutputFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.dto.OutputFavouriteUserItemSubscriberDto;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.enums.ItemContext;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;

import java.util.List;
import java.util.Set;
import java.util.UUID;

/**
 * Service for handling business logic related to favourite user items.
 */
public interface FavouriteUserItemService {
    /**
     * Creates a new favourite user item.
     *
     * @param userId the id of the logged in user
     * @param dto Creation request. Cannot be null.
     * @return Created favourite user item. Never null.
     * @throws WP2DuplicateResourceException if the favourite user item already exists
     */
    @NonNull
    OutputFavouriteUserItemDto create(@NonNull UUID userId, @NonNull InputCreateFavouriteUserItemDto dto);

    /**
     * Retrieves a favourite user item by id.
     *
     * @param userId the id of the logged in user
     * @param favouriteUserItemId the id of the favourite user item to retrieve
     * @return an {@link OutputFavouriteUserItemDto} representing the favourite user item details
     * @throws WP2ResourceNotFoundException if no favourite user item is found with the given id
     */
    @NonNull
    OutputFavouriteUserItemDto findById(@NonNull UUID userId, @NonNull UUID favouriteUserItemId);

    /**
     * Deletes the favourite user item by id.
     *
     * @param userId the id of the logged in user
     * @param favouriteUserItemId the id of the favourite user item to delete
     * @throws WP2BusinessException         if delete is not allowed by configuration
     * @throws WP2ResourceNotFoundException if no favourite user item is found with the given id
     */
    void deleteById(@NonNull UUID userId, @NonNull UUID favouriteUserItemId);

    /**
     * Updates only the provided fields of the favourite user item.
     *
     * @param userId the id of the logged in user
     * @param favouriteUserItemId the id of the favourite user item to update
     * @param dto Patch request. Cannot be null.
     * @throws WP2ResourceNotFoundException if no favourite user item is found with the given id
     */
    @NonNull
    OutputFavouriteUserItemDto patchById(@NonNull UUID userId, @NonNull UUID favouriteUserItemId, @NonNull InputPatchFavouriteUserItemDto dto);

    /**
     * Retrieves a page of results of matching favourite user item for the given context.
     *
     * @param userId the id of the logged in user
     * @param context  the context to match
     * @param subContext the sub context to match
     * @param pageable the pagination information
     * @return a page of {@link OutputFavouriteUserItemDto} representing the matching favourite user items
     */
    @NonNull
    Page<OutputFavouriteUserItemDto> findByContext(@NonNull UUID userId, @NonNull ItemContext context, String subContext, @NonNull Pageable pageable);

    /**
     * Retrieves a list of results of matching favourite user item for the given context and item ids.
     *
     * @param userId the id of the logged in user
     * @param context the context to match
     * @param subContext the sub context to match
     * @param itemIds the item ids to match
     * @return a list of {@link OutputFavouriteUserItemDto} representing the matching favourite user items
     */
    @NonNull
    List<OutputFavouriteUserItemDto> findByContextAndItemIds(@NonNull UUID userId, @NonNull ItemContext context,
                                                             String subContext, @NonNull Set<String> itemIds);

    /**
     * Retrieves a list of results of matching favourite user item subscribers for the given context and item ids.
     *
     * @param context    the context to match
     * @param subContext the sub context to match
     * @param itemId     the item id to match
     * @param pageable   the pagination information
     * @return a page of {@link OutputFavouriteUserItemSubscriberDto} representing the matching favourite user items
     */
    @NonNull
    Page<OutputFavouriteUserItemSubscriberDto> findSubscribers(@NonNull ItemContext context, String subContext,
                                                               @NonNull String itemId, @NonNull Pageable pageable);

    /**
     * Deletes favourite user item for all users for the given context and item id.
     *
     * @param context    the context to match
     * @param subContext the sub context to match
     * @param itemId     the item id to match
     * @return a list of {@link OutputFavouriteUserItemDto} representing the deleted favourite user items
     */
    List<OutputFavouriteUserItemDto> deleteByContextAndItemId(@NonNull ItemContext context, String subContext, @NonNull String itemId);
}
