package com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging;

import com.fasterxml.jackson.core.type.TypeReference;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.config.FavouriteUserBsProperties;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.config.MessagingInstitutionalPageBsProperties;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.enums.ItemContext;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.dto.InstitutionalPageStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.service.FavouriteUserItemService;
import com.finconsgroup.itserr.messaging.consumer.CloudEventConsumer;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import io.cloudevents.CloudEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

/**
 * Concrete implementation of {@link CloudEventConsumer} for handling institutional page related events.
 */
@Component("institutionalPageEventConsumer")
@Slf4j
public class InstitutionalPageEventConsumer extends CloudEventConsumer {

    private static final String OPERATION_TYPE_CREATE = "create";
    private static final String OPERATION_TYPE_DELETE = "delete";

    private final FavouriteUserItemService favouriteUserItemService;

    /**
     * Constructs the InstitutionalPageEventConsumer
     *
     * @param favouriteUserBsConfigurationProperties the application configuration properties
     * @param favouriteUserItemService               favourite user item service
     */
    public InstitutionalPageEventConsumer(FavouriteUserBsProperties favouriteUserBsConfigurationProperties,
                                          FavouriteUserItemService favouriteUserItemService) {
        MessagingInstitutionalPageBsProperties messagingInstitutionalPageBsProperties =
                favouriteUserBsConfigurationProperties.messaging().getInstitutionalPageBs();
        addHandler(messagingInstitutionalPageBsProperties.getResourceStatusChangeEventType(), new TypeReference<>() {
                },
                this::handleResourceStatusChange);

        this.favouriteUserItemService = favouriteUserItemService;
    }

    private void handleResourceStatusChange(MessagingEventDto<InstitutionalPageStatusChangeNotificationData> eventPayload,
                                            CloudEvent cloudEvent) {
        InstitutionalPageStatusChangeNotificationData data = eventPayload.getAdditionalData();

        // when the public institutional page creation is approved for the first time
        // generate the institutional page created by favourite user event
        if (OPERATION_TYPE_CREATE.equalsIgnoreCase(data.getOperationType())
                && Boolean.TRUE.equals(data.getApproved())
                && Boolean.TRUE.equals(data.getPublished())
                && data.getMaintainer() != null) {
            favouriteUserItemService.publishResourceCreatedByFolloweeEvent(ItemContext.INSTITUTIONAL_PAGE.getId(),
                    data.getMaintainer().toString(), eventPayload);
        }

        if (OPERATION_TYPE_DELETE.equalsIgnoreCase(data.getOperationType())
                && Boolean.TRUE.equals(data.getApproved())) {
            // when the public institutional page deletion is approved
            // then delete all the favourite items related to the deleted institutional page
            favouriteUserItemService.deleteByContextAndItemId(ItemContext.INSTITUTIONAL_PAGE, null,
                    eventPayload.getId(), eventPayload.getUser());
        }
    }

}