package com.finconsgroup.itserr.marketplace.favourite.user.bs.api;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.core.web.utils.OpenApiUtils;
import com.finconsgroup.itserr.marketplace.core.web.validation.annotation.ValidPutIdParameters;
import com.finconsgroup.itserr.marketplace.core.web.validation.annotation.ValidQueryFilter;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.FavouriteUserItemDetail;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.InputCreateFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.InputPatchFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.OutputFavouriteUserItemDetailDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.OutputFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.enums.ItemContext;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotBlank;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import jakarta.validation.constraints.Size;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.List;
import java.util.UUID;

/**
 * APIs for managing and querying the favourite user items.
 */
@Tag(name = "Favourite User Items", description = "Operations related to favourite user items")
@SecurityRequirement(name = "BearerAuth")
@ApiResponses(value = {
    @ApiResponse(responseCode = "401", ref = OpenApiUtils.RESPONSE_UNAUTHORIZED),
    @ApiResponse(responseCode = "500", ref = OpenApiUtils.RESPONSE_INTERNAL_SERVER_ERROR)
})
@Validated
public interface FavouriteUserItemApi {

    /**
     * Creates a new favourite user item
     *
     * @param dto the input data transfer object containing favourite user item details
     * @return the created {@link OutputFavouriteUserItemDto} and HTTP status 201 (CREATED)
     */
    @Operation(
        summary = "Create new favourite user item",
        description = "Create a new favourite user item for the given context based on" +
            "the provided dto. It also performs validations and returns errors in case of validation failures"
    )
    @ApiResponses(value = {
        @ApiResponse(responseCode = "201", description = "OK"),
        @ApiResponse(responseCode = "400", ref = OpenApiUtils.RESPONSE_BAD_REQUEST)
    })
    @ResponseStatus(HttpStatus.CREATED)
    @PostMapping(value = "/favourite-user/items", consumes = MediaType.APPLICATION_JSON_VALUE,
        produces = MediaType.APPLICATION_JSON_VALUE)
    OutputFavouriteUserItemDto create(@Valid @RequestBody InputCreateFavouriteUserItemDto dto);

    /**
     * Fetch favourite user item for the provided id.
     *
     * @param favouriteUserItemId the id of the favourite user item to fetch
     * @return the found {@link OutputFavouriteUserItemDto} and HTTP status 200 (Ok)
     */
    @Operation(
        summary = "Fetches favourite user item for the provided id",
        description = "Fetches a favourite user item using the provided id."
            + "It returns not found response in case if the id does not exist"
    )
    @ApiResponses(value = {
        @ApiResponse(responseCode = "200", description = "OK"),
        @ApiResponse(responseCode = "404", ref = OpenApiUtils.RESPONSE_NOT_FOUND)
    })
    @ResponseStatus(HttpStatus.OK)
    @GetMapping(value = "/favourite-user/items/{favouriteUserItemId}", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputFavouriteUserItemDetailDto<?> get(@PathVariable("favouriteUserItemId") UUID favouriteUserItemId);

    /**
     * Delete favourite user item for provided id.
     *
     * @param favouriteUserItemId the id of the favourite user item to delete
     *                Returns HTTP status 204 (No Content), if the favourite user item is deleted successfully
     */
    @Operation(
        summary = "Deletes favourite user item for the provided id",
        description = "Deletes a favourite user item using the provided id."
            + "It returns not found response in case if the id does not exist"
    )
    @ApiResponses(value = {
        @ApiResponse(responseCode = "204", description = "No Content"),
        @ApiResponse(responseCode = "404", ref = OpenApiUtils.RESPONSE_NOT_FOUND)
    })
    @ResponseStatus(HttpStatus.NO_CONTENT)
    @DeleteMapping("/favourite-user/items/{favouriteUserItemId}")
    void delete(@PathVariable("favouriteUserItemId") UUID favouriteUserItemId);

    /**
     * Updates only the provided fields of favourite user item i.e. patch operation.
     *
     * @param dto the input data transfer object containing favourite user item details to update
     * @return the updated {@link OutputFavouriteUserItemDto} and HTTP status 200 (OK)
     */
    @Operation(
        summary = "Updates an existing favourite user item",
        description = "Updates only the provided fields of favourite user item based on " +
            "the provided dto. It also performs validations and returns errors in case of validation failures"
    )
    @ApiResponses(value = {
        @ApiResponse(responseCode = "200", description = "OK"),
        @ApiResponse(responseCode = "400", ref = OpenApiUtils.RESPONSE_BAD_REQUEST),
        @ApiResponse(responseCode = "404", ref = OpenApiUtils.RESPONSE_NOT_FOUND)
    })
    @ResponseStatus(HttpStatus.OK)
    @PatchMapping(value = "/favourite-user/items/{favouriteUserItemId}", produces = MediaType.APPLICATION_JSON_VALUE,
        consumes = MediaType.APPLICATION_JSON_VALUE)
    @ValidPutIdParameters(pathIdParam = "favouriteUserItemId")
    OutputFavouriteUserItemDto patch(@PathVariable("favouriteUserItemId") UUID favouriteUserItemId,
                                     @Valid @RequestBody InputPatchFavouriteUserItemDto dto);

    /**
     * Retrieves a paginated list of all favourite user items matching the provided context.
     *
     * @param context    the {@link ItemContext} to search favourite user items for
     * @param subContext the subContext to search favourite user items for (optional)
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of favourite user item per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @param filters    the additional filters to apply in the format key1=value1,value2&key2=value3,value4
     * @return ResponseEntity containing a page of item detail dtos and HTTP status 200 (OK)
     */
    @Operation(
        summary = "Retrieve all favourite user items matching the provided context",
        parameters = {
            @Parameter(name = "context", description = "The context to search favourite user items for"),
            @Parameter(name = "subContext", description = "The sub context to search favourite user items for"),
            @Parameter(name = "pageNumber", description = "The page index, starting from 0"),
            @Parameter(name = "pageSize", description = "The number of items per page"),
            @Parameter(name = "sort", description = "The field by which to sort the results"),
            @Parameter(name = "direction", description = "The sorting direction: ASC or DESC"),
            @Parameter(name = "filters", description = "The additional filters to apply in the format key1=value1,value2&key2=value3,value4")
        },
        responses = {
            @ApiResponse(responseCode = "200", description = "OK"),
            @ApiResponse(responseCode = "400", ref = OpenApiUtils.RESPONSE_BAD_REQUEST)
        }
    )
    @ResponseStatus(HttpStatus.OK)
    @GetMapping(value = "/favourite-user/items", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputPageDto<OutputFavouriteUserItemDetailDto<FavouriteUserItemDetail>> findByContext(
        @RequestParam(name = "context") ItemContext context,
        @RequestParam(name = "subContext", required = false) String subContext,
        @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
        @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
        @RequestParam(name = "sort", defaultValue = "id", required = false) String sort,
        @RequestParam(name = "direction", defaultValue = "ASC", required = false) SortDirection direction,
        @RequestParam(name = "filters", required = false) @ValidQueryFilter String filters
    );

    /**
     * Retrieves a paginated list of all favourite user items matching the provided context.
     *
     * @param context           the {@link ItemContext} to search favourite user items for
     * @param subContext        the subContext to search favourite user items for (optional)
     * @param itemIds           the delimiter (default ',') separated list of item ids
     * @param itemIdSeparator   the separator for item ids (default if ",")
     * @return ResponseEntity containing a page of OutputFavouriteUserItemDto and HTTP status 200 (OK)
     */
    @Operation(
        summary = "Retrieve all favourite user items matching the provided context",
        parameters = {
            @Parameter(name = "context", description = "The context to search favourite user items for"),
            @Parameter(name = "subContext", description = "The sub context to search favourite user items for"),
            @Parameter(name = "itemIds", description = "The items ids separated by a separator"),
            @Parameter(name = "itemIdSeparator", description = "The item id separator, default if comma i.e ','")
        },
        responses = {
            @ApiResponse(responseCode = "200", description = "OK"),
            @ApiResponse(responseCode = "400", ref = OpenApiUtils.RESPONSE_BAD_REQUEST)
        }
    )
    @ResponseStatus(HttpStatus.OK)
    @GetMapping(value = "/favourite-user/items-by-item-ids", produces = MediaType.APPLICATION_JSON_VALUE)
    List<OutputFavouriteUserItemDto> findByContextAndItemIds(
        @RequestParam(name = "context") ItemContext context,
        @RequestParam(name = "subContext", required = false) String subContext,
        @NotBlank @RequestParam(name = "itemIds") String itemIds,
        @Size(min = 1) @RequestParam(name = "itemIdSeparator", required = false, defaultValue = ",") String itemIdSeparator
    );

}
