package com.finconsgroup.itserr.marketplace.favouritesearch.dm.service.impl;

import com.finconsgroup.itserr.marketplace.favouritesearch.dm.dto.InputCreateFavouriteSearchDto;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.dto.OutputFavouriteSearchDto;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.entity.ArchivedFavouriteSearchEntity;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.entity.FavouriteSearchEntity;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.enums.SearchContext;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.exception.FavouriteSearchNotFoundException;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.mapper.ArchivedEntityMapper;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.mapper.FavouriteSearchMapper;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.repository.ArchivedFavouriteSearchRepository;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.repository.FavouriteSearchRepository;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.service.FavouriteSearchService;
import io.micrometer.common.util.StringUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.UUID;

/**
 * Default implementation of {@link FavouriteSearchService}
 * to perform operations related to favouriteSearch resources
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultFavouriteSearchService implements FavouriteSearchService {

    private final FavouriteSearchRepository favouriteSearchRepository;
    private final FavouriteSearchMapper favouriteSearchMapper;

    private final ArchivedFavouriteSearchRepository archivedFavouriteSearchRepository;
    private final ArchivedEntityMapper archivedEntityMapper;

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputFavouriteSearchDto create(
            @NonNull UUID userId,
            @NonNull InputCreateFavouriteSearchDto inputCreateFavouriteSearchDto
    ) {
        // For now no error is thrown if a favouriteSearch with the same name already exists
        FavouriteSearchEntity favouriteSearchEntity = favouriteSearchMapper.toEntity(inputCreateFavouriteSearchDto, userId);
        FavouriteSearchEntity savedFavouriteSearchEntity = favouriteSearchRepository.saveAndFlush(favouriteSearchEntity);
        return favouriteSearchMapper.toDto(savedFavouriteSearchEntity);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputFavouriteSearchDto> findAll(
            @NonNull UUID userId,
            SearchContext context,
            @NonNull String searchText,
            @NonNull Pageable pageable
    ) {
        Page<FavouriteSearchEntity> favouriteSearchEntityPage;

        boolean hasSearchText = StringUtils.isNotBlank(searchText);
        boolean hasContext = context != null;

        if (hasSearchText && hasContext) {
            favouriteSearchEntityPage = favouriteSearchRepository
                    .findAllByUserIdAndContextAndSearchTextContainingIgnoreCaseOrUserIdAndContextAndNameContainingIgnoreCase(
                            userId,      // userId for searchText
                            context,     // context for searchText
                            searchText,  // value for searchText
                            userId,      // userId for name
                            context,     // context for searchText
                            searchText,  // value for name
                            pageable);
        } else if (hasSearchText) {
            favouriteSearchEntityPage = favouriteSearchRepository
                    .findAllByUserIdAndSearchTextContainingIgnoreCaseOrUserIdAndNameContainingIgnoreCase(
                            userId,     // userId for searchText
                            searchText, // value for searchText
                            userId,     // userId for name
                            searchText, // value for name
                            pageable);
        } else if (hasContext) {
            favouriteSearchEntityPage = favouriteSearchRepository
                    .findByUserIdAndContext(userId, context, pageable);
        } else {
            favouriteSearchEntityPage = favouriteSearchRepository
                    .findAllByUserId(userId, pageable);
        }
        return favouriteSearchEntityPage
                .map(favouriteSearchMapper::toDto);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputFavouriteSearchDto findById(@NonNull UUID userId, @NonNull UUID favouriteSearchId) {
        return favouriteSearchRepository.findByUserIdAndId(userId, favouriteSearchId)
                .map(favouriteSearchMapper::toDto)
                .orElseThrow(() -> new FavouriteSearchNotFoundException(favouriteSearchId));
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void deleteById(
            @NonNull UUID userId,
            @NonNull UUID favouriteSearchId
    ) {
        // retrieve favouriteSearch
        FavouriteSearchEntity favouriteSearchEntity = favouriteSearchRepository.findByUserIdAndId(userId, favouriteSearchId)
                .orElseThrow(() -> new FavouriteSearchNotFoundException(favouriteSearchId));
        // archive and remove the entity
        persistArchivedCopy(favouriteSearchEntity);
        favouriteSearchRepository.delete(favouriteSearchEntity);
    }

    // private

    private void persistArchivedCopy(FavouriteSearchEntity favouriteSearchEntity) {
        ArchivedFavouriteSearchEntity archivedFavouriteSearchEntity = archivedEntityMapper.toArchiveEntity(favouriteSearchEntity);
        archivedFavouriteSearchRepository.saveAndFlush(archivedFavouriteSearchEntity);
    }
}