package com.finconsgroup.itserr.marketplace.favouritesearch.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.converter.SearchContextAttributeConverter;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.enums.SearchContext;
import jakarta.persistence.Column;
import jakarta.persistence.Convert;
import jakarta.persistence.Entity;
import jakarta.persistence.PrePersist;
import jakarta.persistence.PreUpdate;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.UUID;

/**
 * Entity class representing a favouriteSearch in the favourite-search microservice.
 * This entity is mapped to the "favourite_search" table in the database
 * and stores information such as
 * the user ID, the name, the search input, the search filters, creation time, and update time.
 *
 * <p>Example usage:
 * <pre>
 * FavouriteSearchEntity favouriteSearch = FavouriteSearchEntity.builder()
 * 	   .name("My favourite search")
 *     .userId(userId)
 *     .searchText("Godfath")
 *     .searchFilters("{type:dataset}")
 *     .build();
 * </pre>
 * </p>
 */
@Entity
@Table(name = "favourite_search")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class FavouriteSearchEntity extends AbstractUUIDEntity {
    /**
     * The name of the favourite search.
     */
    @Column(name = "name", nullable = false)
    private String name;

    /**
     * The ID of the associated user.
     */
    @Column(name = "user_id", nullable = false, updatable = false)
    private UUID userId;

    /**
     * The context of the favourite search.
     */
    @Column(name = "context", nullable = false, updatable = false)
    @Convert(converter = SearchContextAttributeConverter.class)
    private SearchContext context;

    /**
     * The text used to perform the search.
     */
    @Column(name = "search_text", nullable = false, updatable = false)
    private String searchText;

    /**
     * Raw filters of the search provided by the frontend
     */
    @Column(name = "search_filters")
    private String searchFilters;

    /**
     * The timestamp when the favouriteSearch was created.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private ZonedDateTime creationTime;

    /**
     * The timestamp when the favouriteSearch was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private ZonedDateTime updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     * </p>
     */
    @Version
    private long version;

    /**
     * Sets both {@code creationTime} and {@code updateTime} before the entity is persisted.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PrePersist
    public void onCreate() {
        ZonedDateTime now = ZonedDateTime.now(ZoneOffset.UTC).truncatedTo(ChronoUnit.MICROS);
        creationTime = now;
        updateTime = now;
    }

    /**
     * Updates {@code updateTime} just before the entity is updated.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PreUpdate
    public void onUpdate() {
        updateTime = ZonedDateTime.now(ZoneOffset.UTC).truncatedTo(ChronoUnit.MICROS);
    }

}
