package com.finconsgroup.itserr.marketplace.event.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.ManyToOne;
import jakarta.persistence.PreUpdate;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.event.dm.util.DomainConstants.MEDIUM_TEXT_LENGTH;

/**
 * Entity class representing a event conductor in the event system.
 */
@SuppressWarnings("DefaultAnnotationParam")
@Entity
@Table(name = "event_conductor")
@SuperBuilder
@Getter
@Setter
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class EventConductorEntity extends AbstractUUIDEntity {

    /**
     * Association to the related event
     */
    @ManyToOne(fetch = FetchType.LAZY, optional = false)
    @JoinColumn(name = "event_id", updatable = false)
    private EventEntity event;

    /**
     * The ID of the related event.
     */
    @Column(name = "event_id", insertable = false, updatable = false)
    private UUID eventId;

    /**
     * The first name.
     */
    @Column(name = "first_name", nullable = false, length = MEDIUM_TEXT_LENGTH)
    private String firstName;

    /**
     * The last name.
     */
    @Column(name = "last_name", nullable = false, length = MEDIUM_TEXT_LENGTH)
    private String lastName;

    /**
     * The email.
     */
    @Column(name = "email", length = MEDIUM_TEXT_LENGTH)
    private String email;

    /**
     * The orc id.
     */
    @Column(name = "orcid", length = MEDIUM_TEXT_LENGTH)
    private String orcid;

    /**
     * The timestamp when the event was created.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    /**
     * The timestamp when the event was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     */
    @Version
    @Column(name = "version", nullable = false)
    private long version;

    /**
     * The ordering column.
     */
    @Column(name = "conductor_order", nullable = false)
    private long conductorOrder;

    @Override
    public void prePersist() {
        super.prePersist();
        Instant now = Instant.now();
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
    }

    /**
     * Updates {@code updateTime} just before the entity is updated.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PreUpdate
    public void onUpdate() {
        updateTime = Instant.now();
    }

}
