package com.finconsgroup.itserr.marketplace.event.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.CascadeType;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.OneToMany;
import jakarta.persistence.Table;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.ToString;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.time.LocalTime;
import java.util.List;

import static com.finconsgroup.itserr.marketplace.event.dm.util.DomainConstants.LINK_TEXT_LENGTH;
import static com.finconsgroup.itserr.marketplace.event.dm.util.DomainConstants.MEDIUM_TEXT_LENGTH;
import static com.finconsgroup.itserr.marketplace.event.dm.util.DomainConstants.SHORT_TEXT_LENGTH;

/**
 * Entity class representing an archived program in the event system.
 */
@SuppressWarnings("DefaultAnnotationParam")
@Entity
@Table(name = "archived_program")
@SuperBuilder
@Getter
@Setter
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class ArchivedProgramEntity extends AbstractUUIDEntity {

    /**
     * The title.
     */
    @Column(name = "title", nullable = false, length = MEDIUM_TEXT_LENGTH)
    private String title;

    /**
     * The description.
     */
    @Column(name = "description", length = MEDIUM_TEXT_LENGTH)
    private String description;

    /**
     * The program link
     */
    @Column(name = "program_link", length = LINK_TEXT_LENGTH)
    private String programLink;

    /**
     * The place.
     */
    @Column(name = "place", length = MEDIUM_TEXT_LENGTH)
    private String place;

    /**
     * The street.
     */
    @Column(name = "street", length = MEDIUM_TEXT_LENGTH)
    private String street;

    /**
     * The house number.
     */
    @Column(name = "house_number", length = SHORT_TEXT_LENGTH)
    private String houseNumber;

    /**
     * Start Time of the schedule.
     */
    @Column(name = "start_time", nullable = false)
    private LocalTime startTime;

    /**
     * End Time of the schedule.
     */
    @Column(name = "end_time", nullable = false)
    private LocalTime endTime;

    /**
     * The maximum number of participants allowed for the program.
     */
    @Column(name = "max_participants")
    private Integer maxParticipants;

    /**
     * The number of participants subscribed for the program.
     */
    @Column(name = "subscribed_participants")
    private Integer subscribedParticipantsCount;

    /**
     * The subscribed participants related to the program.
     */
    @OneToMany(fetch = FetchType.LAZY, cascade = CascadeType.ALL, orphanRemoval = true, mappedBy = "program")
    @ToString.Exclude
    private List<ArchivedProgramSubscribedParticipantEntity> subscribedParticipants;

    /**
     * The program conductors related to the program.
     */
    @OneToMany(fetch = FetchType.LAZY, cascade = CascadeType.ALL, orphanRemoval = true)
    @JoinColumn(name = "program_id", nullable = false, updatable = false)
    @ToString.Exclude
    private List<ArchivedProgramConductorEntity> programConductors;

    /**
     * The timestamp when the event was created.
     */
    @Column(name = "creation_time", nullable = false)
    private Instant creationTime;

    /**
     * The timestamp when the event was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     */
    @Column(name = "version", nullable = false)
    private long version;

    /**
     * The ordering column.
     */
    @Column(name = "program_order", nullable = false)
    private long programOrder;

    @Override
    public void prePersist() {
        super.prePersist();
        Instant now = Instant.now();
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
    }

}
