package com.finconsgroup.itserr.marketplace.event.bs.component;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.event.bs.bean.EventApplicationEvent;
import com.finconsgroup.itserr.marketplace.event.bs.dto.OutputEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.enums.MessagingEventType;
import com.finconsgroup.itserr.marketplace.event.bs.mapper.EventMapper;
import com.finconsgroup.itserr.marketplace.event.bs.messaging.EventProducer;
import com.finconsgroup.itserr.marketplace.event.bs.messaging.ResourceProducer;
import com.finconsgroup.itserr.marketplace.event.bs.messaging.dto.EventCreatedMessagingAdditionalDataDto;
import com.finconsgroup.itserr.marketplace.event.bs.messaging.dto.EventMessagingAdditionalDataDto;
import com.finconsgroup.itserr.marketplace.event.bs.messaging.dto.EventSubscriptionMessagingAdditionalDataDto;
import com.finconsgroup.itserr.marketplace.event.bs.messaging.dto.EventUpdateMessagingAdditionalDataDto;
import com.finconsgroup.itserr.marketplace.event.dm.dto.OutputSubscribedParticipantDto;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import com.finconsgroup.itserr.messaging.dto.MessagingEventUserDto;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationListener;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.stream.Collectors;

/**
 * The implementation for {@link ApplicationListener} to listen to Event application events
 * and produce relevant messages.
 */
@Component
@Slf4j
public class MessagingEventApplicationEventListener implements ApplicationListener<EventApplicationEvent> {

    private final EventProducer eventProducer;
    private final ResourceProducer resourceProducer;
    private final EventMapper eventMapper;

    public MessagingEventApplicationEventListener(EventProducer eventProducer,
                                                  ResourceProducer resourceProducer,
                                                  EventMapper eventMapper) {
        this.eventProducer = eventProducer;
        this.resourceProducer = resourceProducer;
        this.eventMapper = eventMapper;
    }

    @Override
    public void onApplicationEvent(@NonNull EventApplicationEvent event) {

        MessagingEventDto<EventMessagingAdditionalDataDto> messagingEventDto =
                eventMapper.toMessagingEventDto(event.getEvent());

        MessagingEventUserDto user = MessagingEventUserDto.builder()
                .id(JwtTokenHolder.getUserId().orElse(null))
                .name(JwtTokenHolder.getName().orElse(null))
                .username(JwtTokenHolder.getPreferredUsername().orElse(null))
                .build();
        messagingEventDto.setUser(user);

        updateEventForType(event, messagingEventDto);
        publishEventForType(event, messagingEventDto);
    }

    private void updateEventForType(EventApplicationEvent applicationEvent, MessagingEventDto<EventMessagingAdditionalDataDto> messagingEventDto) {
        OutputEventDto outputEventDto = applicationEvent.getEvent();
        EventMessagingAdditionalDataDto additionalData = null;

        switch (applicationEvent.getEventType()) {
            case CREATED -> {
                additionalData = createAdditionalData(applicationEvent.getEventType());
                messagingEventDto.setTimestamp(outputEventDto.getCreationTime());
                updateAdditionalDataForCreated(additionalData, outputEventDto);
            }
            case UPDATED -> {
                additionalData = createAdditionalData(applicationEvent.getEventType());
                messagingEventDto.setTimestamp(outputEventDto.getUpdateTime());
                if (outputEventDto.getSubscribedParticipantsCount() != null
                        && outputEventDto.getSubscribedParticipantsCount() > 0) {
                    updateAdditionalDataForUpdated(additionalData, outputEventDto);
                }
            }
            case DELETED -> {
                messagingEventDto.setTimestamp(applicationEvent.getEventTimestamp());
            }
            case REGISTER, UNREGISTER -> {
                additionalData = createAdditionalData(applicationEvent.getEventType());
                messagingEventDto.setTimestamp(outputEventDto.getUpdateTime());
                updateAdditionalDataForSubscription(additionalData, outputEventDto, applicationEvent.getEventType());
            }
        }

        messagingEventDto.setAdditionalData(additionalData);
    }

    private void updateAdditionalDataForUpdated(EventMessagingAdditionalDataDto additionalData, OutputEventDto outputEventDto) {
        if (additionalData instanceof EventUpdateMessagingAdditionalDataDto eventUpdateMessagingAdditionalDataDto) {
            if (outputEventDto.getSubscribedParticipantsCount() != null
            && outputEventDto.getSubscribedParticipantsCount() > 0 ) {
                eventUpdateMessagingAdditionalDataDto.setReceivers(
                        outputEventDto.getSubscribedParticipants().stream()
                                .map(OutputSubscribedParticipantDto::getUserId)
                                .collect(Collectors.toSet())
                );
            }
        }
    }

    private EventMessagingAdditionalDataDto createAdditionalData(MessagingEventType eventType) {
        return switch (eventType) {
            case CREATED -> new EventCreatedMessagingAdditionalDataDto();
            case REGISTER, UNREGISTER -> new EventSubscriptionMessagingAdditionalDataDto();
            case UPDATED -> new EventUpdateMessagingAdditionalDataDto();
            default -> new EventMessagingAdditionalDataDto();
        };
    }

    private void updateAdditionalDataForCreated(EventMessagingAdditionalDataDto additionalData,
                                                OutputEventDto outputEventDto) {
        if (additionalData instanceof EventCreatedMessagingAdditionalDataDto createdAdditionalDataDto) {
            createdAdditionalDataDto.setContent(outputEventDto.getContent());
            if (outputEventDto.getImage() != null) {
                createdAdditionalDataDto.setImageUrl(outputEventDto.getImage().getUrl());
            }
        }
    }

    private void updateAdditionalDataForSubscription(EventMessagingAdditionalDataDto additionalData,
                                                     OutputEventDto outputEventDto,
                                                     MessagingEventType eventType) {
        if (additionalData instanceof EventSubscriptionMessagingAdditionalDataDto subscriptionAdditionalDataDto) {
            subscriptionAdditionalDataDto.setNotifyUserIds(List.of(outputEventDto.getMaintainer().getId().toString()));
            subscriptionAdditionalDataDto.setSubscribed(eventType == MessagingEventType.REGISTER);
            subscriptionAdditionalDataDto.setStartDate(outputEventDto.getStartDate());
        }
    }

    private void publishEventForType(EventApplicationEvent applicationEvent, MessagingEventDto<?> messagingEventDto) {
        switch (applicationEvent.getEventType()) {
            case CREATED -> {
                resourceProducer.publishCreatedResource(applicationEvent.getEvent());
                eventProducer.publishCreateEvent(messagingEventDto);
            }
            case UPDATED -> {
                resourceProducer.publishUpdatedResource(applicationEvent.getEvent());
                eventProducer.publishUpdateEvent(messagingEventDto);
            }
            case DELETED -> {
                // publish to delete resource from global search
                resourceProducer.publishDeletedResource(applicationEvent.getEvent());
                eventProducer.publishDeleteEvent(messagingEventDto);
            }
            case REGISTER, UNREGISTER -> {
                // as subscription events update the subscribed participants,
                // we also need to publish the update resource message
                resourceProducer.publishUpdatedResource(applicationEvent.getEvent());
                eventProducer.publishSubscriptionEvent(messagingEventDto);
            }
        }
    }

}
