package com.finconsgroup.itserr.marketplace.event.bs.messaging.impl;


import com.finconsgroup.itserr.marketplace.event.bs.configuration.properties.EventBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.event.bs.dto.OutputEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.messaging.ResourceProducer;
import com.finconsgroup.itserr.messaging.exception.WP2MessagingException;
import com.finconsgroup.itserr.messaging.producer.MessageProducer;
import com.finconsgroup.itserr.messaging.producer.ProducerRegistry;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.Map;

import static com.finconsgroup.itserr.marketplace.event.bs.util.Constants.SERVICE_FILTER_KEY;
import static com.finconsgroup.itserr.marketplace.event.bs.util.Constants.SERVICE_FILTER_VALUE;


/**
 * Default implementation of {@link ResourceProducer} that publishes
 * resource events on approval for an event.
 */
@Component
@ConditionalOnProperty(name = "messaging.enabled", havingValue = "true")
@Slf4j
public class DefaultResourceProducer implements ResourceProducer {

    private final MessageProducer messageProducer;

    private final EventBsConfigurationProperties.Messaging messagingProperties;

    /**
     * Constructs the DefaultResourceProducer.
     *
     * @param bsConfigurationProperties application-specific configuration properties
     * @param producerRegistry          the registry providing message producers
     */
    public DefaultResourceProducer(
            EventBsConfigurationProperties bsConfigurationProperties,
            ProducerRegistry producerRegistry
    ) {
        this.messagingProperties = bsConfigurationProperties.getMessaging();
        this.messageProducer = producerRegistry.getMessageProducer(this.messagingProperties.getProducerName());
    }

    @Override
    public void publishCreatedResource(@NotNull OutputEventDto resource) {
        publish(resource, messagingProperties.getResourceCreatedType());
    }

    @Override
    public void publishUpdatedResource(@NotNull OutputEventDto resource) {
        publish(resource, messagingProperties.getResourceUpdatedType());
    }

    @Override
    public void publishDeletedResource(@NotNull OutputEventDto resource) {
        publish(resource, messagingProperties.getResourceDeletedType());
    }

    /**
     * Sends a message to the messaging system.
     *
     * @param resource  the payload of the message
     * @param eventType the type of event to publish
     * @throws WP2MessagingException if message publishing fails
     */
    private void publish(@NonNull OutputEventDto resource, @NonNull String eventType) {
        try {
            Map<String, OutputEventDto> messageMap = Map.of(resource.getId().toString(), resource);
            Map<String, String> filterProperties = Map.of(SERVICE_FILTER_KEY, SERVICE_FILTER_VALUE);
            log.debug("Sending message, messageMap: {}, eventType: {}, source: {}, filterProperties: {}",
                    messageMap, eventType, messagingProperties.getSource(), filterProperties);
            messageProducer.publishEvent(messageMap, eventType, messagingProperties.getSource(), filterProperties);
            log.info("Successfully published message for resource with id: {}", resource.getId());
        } catch (Exception e) {
            String errorMessage = "Error occurred while sending message: %s".formatted(e.getMessage());
            throw new WP2MessagingException(errorMessage);
        }
    }
}
