package com.finconsgroup.itserr.marketplace.event.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.event.bs.client.WorkspaceBsClient;
import com.finconsgroup.itserr.marketplace.event.bs.client.dto.workspace.FolderDto;
import com.finconsgroup.itserr.marketplace.event.bs.client.dto.workspace.InputFolderCreateDto;
import com.finconsgroup.itserr.marketplace.event.bs.client.dto.workspace.OutputPublicImageWSDto;
import com.finconsgroup.itserr.marketplace.event.bs.client.dto.workspace.WorkspaceDto;
import com.finconsgroup.itserr.marketplace.event.bs.configuration.properties.EventBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.event.bs.dto.OutputPublicImageDto;
import com.finconsgroup.itserr.marketplace.event.bs.service.ImageService;
import feign.FeignException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

/**
 * Default implementation of {@link ImageService} to perform operations related to images
 */
@Service
@Slf4j
public class DefaultImageService implements ImageService {

    private final WorkspaceBsClient workspaceBsClient;
    private final EventBsConfigurationProperties eventBsConfigurationProperties;

    public DefaultImageService(WorkspaceBsClient workspaceBsClient,
                               EventBsConfigurationProperties eventBsConfigurationProperties) {
        this.workspaceBsClient = workspaceBsClient;
        this.eventBsConfigurationProperties = eventBsConfigurationProperties;
    }

    @NonNull
    @Override
    public OutputPublicImageDto createPublicImage(@NonNull MultipartFile file) {
        String folderId = getOrCreateImagesFolder();
        OutputPublicImageWSDto wsOutputPublicImageWSDto = workspaceBsClient.createPublicImage(folderId, file);
        return mapWsOutputPublicImageDto(wsOutputPublicImageWSDto);
    }

    @NonNull
    @Override
    public OutputPublicImageDto updatePublicImage(@NonNull String id, @NonNull MultipartFile file) {
        OutputPublicImageWSDto wsOutputPublicImageWSDto = workspaceBsClient.updatePublicImage(id, file);
        return mapWsOutputPublicImageDto(wsOutputPublicImageWSDto);
    }

    /*
     * Creates hidden folder to upload images related to events in user's workspace,
     * if it does not already exist.
     */
    private String getOrCreateImagesFolder() {
        WorkspaceDto workspace = workspaceBsClient.getWorkspace();
        String imagesFolderName = eventBsConfigurationProperties.getImagesFolderName();
        try {
            FolderDto folder = workspaceBsClient.getFolderByParentFolderIdAndName(workspace.getId(), imagesFolderName);
            return folder.getId();
        } catch (FeignException.NotFound | WP2ResourceNotFoundException e) {
            InputFolderCreateDto inputFolderCreateDto = InputFolderCreateDto.builder()
                    .name(imagesFolderName)
                    .description(imagesFolderName)
                    .hidden(true)
                    .build();
            return workspaceBsClient.createFolderByFolderId(workspace.getId(), inputFolderCreateDto);
        }
    }

    /*
     * Maps the OutputPublicImageDto from Workspace BS to Event BS's OutputPublicImageDto
     */
    private static OutputPublicImageDto mapWsOutputPublicImageDto(OutputPublicImageWSDto wsOutputPublicImageWSDto) {
        return OutputPublicImageDto.builder()
                .id(wsOutputPublicImageWSDto.getId())
                .publicLink(wsOutputPublicImageWSDto.getPublicLink())
                .build();
    }
}
