package com.finconsgroup.itserr.marketplace.event.bs.client;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.event.bs.dto.InputCreateEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.InputProgramSubscribedParticipantDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.InputUpdateEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.OutputEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.OutputPublicImageDto;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RequestPart;
import org.springframework.web.multipart.MultipartFile;

import java.util.Set;
import java.util.UUID;

/**
 * Feign client interface for Event Domain Service.
 */
@SuppressWarnings("unused")
public interface EventBsClient {
    /**
     * Retrieves a paginated list of all event.
     *
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of UserProfiles per page (default is 10)
     * @param sort       the field to sort by (default is "startTime")
     * @param direction  the direction of sorting (default is descending)
     * @return a page of {@link OutputEventDto} and HTTP status 200 (OK)
     */
    @GetMapping(value = "/event/events", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputPageDto<OutputEventDto> findAll(
            @RequestParam(name = "associationsToLoad") Set<String> associationsToLoad,
            @RequestParam(name = "pageNumber") int pageNumber,
            @RequestParam(name = "pageSize") int pageSize,
            @RequestParam(name = "sort") String sort,
            @RequestParam(name = "direction") SortDirection direction
    );

    /**
     * Retrieves event by id.
     *
     * @param id the id of the event to retrieve
     * @return the found {@link OutputEventDto} and HTTP status 200 (OK)
     */
    @GetMapping(value = "/event/events/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputEventDto findById(@PathVariable("id") UUID id);

    /**
     * Creates new event.
     *
     * @param inputCreateEventDto the input data transfer object containing event details
     */
    @PostMapping(
            value = "/event/events",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    OutputEventDto createEvent(@RequestBody InputCreateEventDto inputCreateEventDto);

    /**
     * Updates existing event by id.
     *
     * @param id                  the id of the event to update
     * @param inputUpdateEventDto the input data transfer object containing updated event details
     * @return the updated {@link OutputEventDto} and HTTP status 200 (OK)
     */
    @PutMapping(
            value = "/event/events/{id}",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    OutputEventDto updateEventById(
            @PathVariable("id") UUID id,
            @RequestBody InputUpdateEventDto inputUpdateEventDto
    );

    /**
     * Deletes event by id.
     *
     * @param id the id of the event to delete
     */
    @DeleteMapping(value = "/event/events/{id}")
    void deleteEventById(@PathVariable("id") UUID id);

    /**
     * Creates an image file, creating a new version of the document.
     *
     * @param file The new image file.
     * @return Created image.
     */
    @PostMapping(path = "/event/images", consumes = MediaType.MULTIPART_FORM_DATA_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    OutputPublicImageDto createPublicImage(@RequestPart MultipartFile file);

    /**
     * Replaces an image file, creating a new version of the document.
     *
     * @param id   id of the image to be updated.
     * @param file The new image file.
     * @return Updated image.
     */
    @PutMapping(path = "/event/images/{id}", consumes = MediaType.MULTIPART_FORM_DATA_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    OutputPublicImageDto updatePublicImage(@PathVariable String id, @RequestPart MultipartFile file);

    /**
     * Register a user to an event.
     */
    @PostMapping(
            value = "/event/events/{id}/register",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    OutputEventDto register(@PathVariable("id") UUID id);

    /**
     * Unregister a user to an event.
     */
    @DeleteMapping(
            value = "/event/events/{id}/register",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    OutputEventDto unregister(@PathVariable("id") UUID id);

    /**
     * Register a user to a program.
     */
    @PostMapping(
            value = "/event/events/{eventId}/programs/{programId}/register",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    OutputEventDto registerProgram(@PathVariable("eventId") UUID eventId,
                                   @PathVariable("programId") UUID programId,
                                   @RequestBody InputProgramSubscribedParticipantDto inputProgramSubscribedParticipantDto);

    /**
     * Unregister a user to a program.
     */
    @DeleteMapping(
            value = "/event/events/{eventId}/programs/{programId}/register",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    OutputEventDto unregisterProgram(@PathVariable("eventId") UUID eventId, @PathVariable("programId") UUID programId);
}
