package com.finconsgroup.itserr.marketplace.discussion.dm.api;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import com.finconsgroup.itserr.marketplace.discussion.dm.dto.DiscussionDTO;
import com.finconsgroup.itserr.marketplace.discussion.dm.dto.InputUpdateDiscussionDto;
import com.finconsgroup.itserr.marketplace.discussion.dm.dto.ThreadDTO;
import com.finconsgroup.itserr.marketplace.discussion.dm.enums.ReactionType;
import io.swagger.v3.oas.annotations.Operation;

import java.time.Instant;
import java.util.UUID;

import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.parameters.RequestBody;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

/**
 * REST API interface for managing discussions, threads, and reactions within the ITSERR marketplace.
 * <p>
 * Provides endpoints for:
 * <ul>
 *     <li>Creating, retrieving, updating, and deleting discussions</li>
 *     <li>Managing threads within discussions</li>
 *     <li>Adding and removing reactions to discussions and threads</li>
 * </ul>
 * <p>
 * All endpoints require authentication via {@code BearerAuth}.
 * Supports pagination, sorting, and filtering for listing discussions and threads.
 * Returns appropriate HTTP status codes for success and error cases.
 */
@Tag(name = "Discussion Management", description = "API for managing discussions, threads, and reactions in the ITSERR marketplace")
@SecurityRequirement(name = "BearerAuth")
public interface DiscussionApi {

    /**
     * Retrieves a discussion by its unique identifier.
     *
     * @param id             the unique identifier of the discussion
     * @param includeThreads whether to include threads in the response (default: true)
     * @return the {@link DiscussionDTO} representing the discussion
     */
    @Operation(summary = "Get discussion by ID", description = "Retrieves a discussion by its unique identifier")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Discussion found successfully",
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = DiscussionDTO.class))),
            @ApiResponse(responseCode = "404", description = "Discussion not found",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "401", description = "Unauthorized",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )})
    })
    @GetMapping("/discussion/{id}")
    @ResponseStatus(HttpStatus.OK)
    DiscussionDTO getDiscussionById(
            @Parameter(description = "The unique identifier of the discussion", required = true)
            @PathVariable String id,
            @Parameter(description = "Include threads in response (default: true)", example = "true")
            @RequestParam(defaultValue = "true", required = false) boolean includeThreads);

    /**
     * Retrieves a discussion associated with the specified resource identifier and type.
     *
     * @param resourceId   the unique identifier of the resource
     * @param resourceType the type of the resource
     * @return the {@link DiscussionDTO} representing the discussion
     */
    @Operation(summary = "Get discussion by resource ID and type",
            description = "Retrieves a discussion associated with the specified resource identifier and type")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Discussion found successfully",
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = DiscussionDTO.class))),
            @ApiResponse(responseCode = "404", description = "Discussion not found",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "401", description = "Unauthorized",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )})
    })
    @GetMapping("/discussion/resource/{resourceId}")
    DiscussionDTO getDiscussionForResource(
            @Parameter(description = "The unique identifier of the resource", required = true)
            @PathVariable String resourceId,
            @Parameter(description = "The type of the resource", in = ParameterIn.QUERY, required = true)
            @RequestParam(name = "resourceType") String resourceType);

    /**
     * Retrieves a paginated list of discussions with sorting and filtering options.
     *
     * @param page           the page number (0-based)
     * @param size           the number of items per page
     * @param sortBy         the field to sort by
     * @param direction      the sort direction (asc or desc)
     * @param visibility     optional filter by visibility
     * @param includeThreads whether to include threads in the response
     * @param resourceType   optional filter by the type of the resource
     * @param createdAfter   optional filter for created after
     * @return a paginated list of {@link DiscussionDTO}
     */
    @Operation(summary = "Get paginated discussions", description = "Retrieves a paginated list of discussions with sorting and filtering options")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Discussions retrieved successfully",
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = Page.class))),
            @ApiResponse(responseCode = "400", description = "Invalid pagination or sorting parameters",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "401", description = "Unauthorized",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )})
    })
    @GetMapping("/discussions")
    Page<DiscussionDTO> getDiscussions(
            @Parameter(description = "Page number (0-based)", example = "0")
            @RequestParam(defaultValue = "0") int page,
            @Parameter(description = "Number of items per page", example = "10")
            @RequestParam(defaultValue = "10") int size,
            @Parameter(description = "Field to sort by", example = "createdAt")
            @RequestParam(defaultValue = "createdAt") String sortBy,
            @Parameter(description = "Sort direction (ASC or DESC)", example = "DESC")
            @RequestParam(defaultValue = "DESC") Sort.Direction direction,
            @Parameter(description = "Filter by visibility (optional)", example = "public")
            @RequestParam(required = false) String visibility,
            @Parameter(description = "Include threads in response (default: true)", example = "true")
            @RequestParam(defaultValue = "true") boolean includeThreads,
            @Parameter(description = "Filter by resource type (optional)", example = "news")
            @RequestParam(required = false) String resourceType,
            @Parameter(description = "Filter by created after (optional)", example = "2025-08-25T11:22:33.123456Z")
            @RequestParam(required = false) Instant createdAfter);

    /**
     * Creates a new discussion.
     *
     * @param discussionDTO the discussion data to create
     * @return 201 Created if successful
     */
    @Operation(summary = "Create new discussion", description = "Creates a new discussion with the provided data")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "201", description = "Discussion created successfully"),
            @ApiResponse(responseCode = "400", description = "Invalid discussion data",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "401", description = "Unauthorized",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "409", description = "Conflict",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}
            )
    })
    @PostMapping("/discussion")
    @ResponseStatus(HttpStatus.CREATED)
    DiscussionDTO createDiscussion(
            @RequestBody(description = "Discussion data to create", required = true,
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = DiscussionDTO.class)))
            @org.springframework.web.bind.annotation.RequestBody DiscussionDTO discussionDTO);

    /**
     * Deletes a discussion by its unique identifier.
     *
     * @param id the unique identifier of the discussion to delete
     * @return 204 No Content if deleted successfully
     */
    @Operation(summary = "Delete discussion", description = "Deletes a discussion by its unique identifier")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "204", description = "Discussion deleted successfully"),
            @ApiResponse(responseCode = "404", description = "Discussion not found",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "401", description = "Unauthorized",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )})
    })
    @DeleteMapping("/discussion/{id}")
    ResponseEntity<Void> deleteDiscussion(
            @Parameter(description = "The unique identifier of the discussion to delete", required = true)
            @PathVariable String id);

    /**
     * Updates a discussion by its unique identifier.
     * This endpoint performs a partial update (PATCH) on the discussion.
     * Only provided fields will be updated.
     *
     * @param discussionId             the unique identifier of the discussion to update
     * @param inputUpdateDiscussionDto the discussion data to update (title, content, url)
     * @param includeThreads           whether to include threads in the response (default: true)
     * @return the updated {@link DiscussionDTO}
     */
    @Operation(summary = "Update discussion", description = "Partially updates a discussion by its unique identifier")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Discussion updated successfully",
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = DiscussionDTO.class))),
            @ApiResponse(responseCode = "404", description = "Discussion not found",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "400", description = "Invalid discussion data",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "401", description = "Unauthorized",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "403", description = "Forbidden - not the owner or not a social resource",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )})
    })
    @PatchMapping("/discussion/{discussionId}")
    @ResponseStatus(HttpStatus.OK)
    DiscussionDTO updateDiscussion(
            @Parameter(description = "The unique identifier of the discussion to update", required = true) @PathVariable UUID discussionId,
            @RequestBody(description = "Discussion data to update", required = true,
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = InputUpdateDiscussionDto.class)))
            @org.springframework.web.bind.annotation.RequestBody InputUpdateDiscussionDto inputUpdateDiscussionDto,
            @Parameter(description = "Include threads in response (default: true)", example = "true")
            @RequestParam(defaultValue = "true", required = false) boolean includeThreads);

    /**
     * Retrieves a paginated list of threads associated with a discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param page         the page number (0-based)
     * @param size         the number of items per page
     * @param sortBy       the field to sort by
     * @param direction    the sort direction (ASC or DESC)
     * @return a paginated list of {@link ThreadDTO}
     */
    @Operation(summary = "Get paginated threads by discussion ID", description = "Retrieves a paginated list of threads associated with a discussion")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Threads retrieved successfully",
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = Page.class))),
            @ApiResponse(responseCode = "400", description = "Invalid pagination or sorting parameters",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "404", description = "Discussion not found",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "401", description = "Unauthorized",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )})
    })
    @GetMapping("/discussion/{discussionId}/threads")
    Page<ThreadDTO> getThreadsByDiscussionId(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @Parameter(description = "Page number (0-based)", example = "0") @RequestParam(defaultValue = "0") int page,
            @Parameter(description = "Number of items per page", example = "10") @RequestParam(defaultValue = "10") int size,
            @Parameter(description = "Field to sort by", example = "createdAt") @RequestParam(defaultValue = "createdAt") String sortBy,
            @Parameter(description = "Sort direction (ASC or DESC)", example = "DESC") @RequestParam(defaultValue = "DESC") Sort.Direction direction);

    /**
     * Adds a new thread to an existing discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param threadDTO    the thread data to add
     * @return 201 Created if successful
     */
    @Operation(summary = "Add thread to discussion", description = "Adds a new thread to an existing discussion")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "201", description = "Thread added successfully"),
            @ApiResponse(responseCode = "400", description = "Invalid thread data",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "404", description = "Discussion not found",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "401", description = "Unauthorized",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )})
    })
    @PostMapping("/discussion/{discussionId}/threads")
    ResponseEntity<ThreadDTO> addThread(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @RequestBody(description = "Thread data to add", required = true,
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = ThreadDTO.class)))
            @org.springframework.web.bind.annotation.RequestBody ThreadDTO threadDTO);

    /**
     * Deletes a specific thread from a discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param threadId     the unique identifier of the thread to delete
     * @return 204 No Content if deleted successfully
     */
    @Operation(summary = "Delete thread from discussion", description = "Deletes a specific thread from a discussion")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "204", description = "Thread deleted successfully"),
            @ApiResponse(responseCode = "404", description = "Discussion or thread not found",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "401", description = "Unauthorized",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )})
    })
    @DeleteMapping("/discussion/{discussionId}/threads/{threadId}")
    ResponseEntity<Void> deleteThread(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @Parameter(description = "The unique identifier of the thread to delete", required = true) @PathVariable String threadId);

    /**
     * Updates a specific thread within a discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param threadId     the unique identifier of the thread to update
     * @param threadDTO    the updated thread data
     * @return 200 if updated successfully
     */
    @Operation(summary = "Update thread in discussion", description = "Updates a specific thread within a discussion")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Thread updated successfully"),
            @ApiResponse(responseCode = "404", description = "Discussion or thread not found",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "400", description = "Invalid thread data",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "401", description = "Unauthorized",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )})
    })
    @org.springframework.web.bind.annotation.PutMapping("/discussion/{discussionId}/threads/{threadId}")
    ResponseEntity<ThreadDTO> updateThread(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @Parameter(description = "The unique identifier of the thread to update", required = true) @PathVariable String threadId,
            @RequestBody(description = "Thread data to update", required = true,
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = ThreadDTO.class)))
            @org.springframework.web.bind.annotation.RequestBody ThreadDTO threadDTO);

    /**
     * Adds a reaction to a discussion by a user.
     *
     * @param discussionId the unique identifier of the discussion
     * @param userId       the unique identifier of the user
     * @param reactionType the type of reaction to add
     * @return 201 Created if successful
     */
    @Operation(summary = "Add reaction to discussion", description = "Adds a reaction to a discussion by a user")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "201", description = "Reaction added successfully"),
            @ApiResponse(responseCode = "404", description = "Discussion not found",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "409", description = "Reaction already exists",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "401", description = "Unauthorized",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )})
    })
    @PostMapping("/discussion/{discussionId}/reactions/{userId}")
    ResponseEntity<Void> addReactionToDiscussion(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @Parameter(description = "The unique identifier of the user", required = true) @PathVariable String userId,
            @Parameter(description = "The type of reaction to add", required = true) @RequestParam ReactionType reactionType);

    /**
     * Removes a reaction from a discussion by a user.
     *
     * @param discussionId the unique identifier of the discussion
     * @param userId       the unique identifier of the user
     * @return 204 No Content if removed successfully
     */
    @Operation(summary = "Remove reaction from discussion", description = "Removes a reaction from a discussion by a user")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "204", description = "Reaction removed successfully"),
            @ApiResponse(responseCode = "404", description = "Discussion or reaction not found",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "401", description = "Unauthorized",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )})
    })
    @DeleteMapping("/discussion/{discussionId}/reactions/{userId}")
    ResponseEntity<Void> removeReactionFromDiscussion(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @Parameter(description = "The unique identifier of the user", required = true) @PathVariable String userId);

    /**
     * Adds a reaction to a specific thread within a discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param threadId     the unique identifier of the thread
     * @param userId       the unique identifier of the user
     * @param reactionType the type of reaction to add
     * @return 201 Created if successful
     */
    @Operation(summary = "Add reaction to thread", description = "Adds a reaction to a specific thread within a discussion")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "201", description = "Reaction added successfully"),
            @ApiResponse(responseCode = "404", description = "Discussion or thread not found",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "409", description = "Reaction already exists",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "401", description = "Unauthorized",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )})
    })
    @PostMapping("/discussion/{discussionId}/threads/{threadId}/reactions/{userId}")
    ResponseEntity<Void> addReactionToThread(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @Parameter(description = "The unique identifier of the thread", required = true) @PathVariable String threadId,
            @Parameter(description = "The unique identifier of the user", required = true) @PathVariable String userId,
            @Parameter(description = "The type of reaction to add", required = true) @RequestParam ReactionType reactionType);

    /**
     * Removes a reaction from a specific thread within a discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param threadId     the unique identifier of the thread
     * @param userId       the unique identifier of the user
     * @return 204 No Content if removed successfully
     */
    @Operation(summary = "Remove reaction from thread", description = "Removes a reaction from a specific thread within a discussion")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "204", description = "Reaction removed successfully"),
            @ApiResponse(responseCode = "404", description = "Discussion, thread, or reaction not found",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "401", description = "Unauthorized",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )}),
            @ApiResponse(responseCode = "403", description = "Forbidden",
                    content = {@Content(
                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                            schema = @Schema(implementation = ErrorResponseDto.class)
                    )})
    })
    @DeleteMapping("/discussion/{discussionId}/threads/{threadId}/reactions/{userId}")
    ResponseEntity<Void> removeReactionFromThread(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @Parameter(description = "The unique identifier of the thread", required = true) @PathVariable String threadId,
            @Parameter(description = "The unique identifier of the user", required = true) @PathVariable String userId);
}
