package com.finconsgroup.itserr.marketplace.discussion.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2AuthorizationException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2DuplicateResourceException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.discussion.dm.bean.DiscussionFilterRequest;
import com.finconsgroup.itserr.marketplace.discussion.dm.dto.DiscussionDTO;
import com.finconsgroup.itserr.marketplace.discussion.dm.dto.InputUpdateDiscussionDto;
import com.finconsgroup.itserr.marketplace.discussion.dm.dto.ThreadDTO;
import com.finconsgroup.itserr.marketplace.discussion.dm.entity.Discussion;
import com.finconsgroup.itserr.marketplace.discussion.dm.entity.DiscussionReaction;
import com.finconsgroup.itserr.marketplace.discussion.dm.entity.Thread;
import com.finconsgroup.itserr.marketplace.discussion.dm.entity.ThreadReaction;
import com.finconsgroup.itserr.marketplace.discussion.dm.enums.ReactionType;
import com.finconsgroup.itserr.marketplace.discussion.dm.mapper.DiscussionMapper;
import com.finconsgroup.itserr.marketplace.discussion.dm.mapper.ThreadMapper;
import com.finconsgroup.itserr.marketplace.discussion.dm.repository.DiscussionReactionRepository;
import com.finconsgroup.itserr.marketplace.discussion.dm.repository.DiscussionRepository;
import com.finconsgroup.itserr.marketplace.discussion.dm.repository.ThreadReactionRepository;
import com.finconsgroup.itserr.marketplace.discussion.dm.repository.ThreadRepository;
import com.finconsgroup.itserr.marketplace.discussion.dm.repository.specification.DiscussionSpecifications;
import com.finconsgroup.itserr.marketplace.discussion.dm.service.DiscussionService;
import lombok.RequiredArgsConstructor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.Instant;
import java.util.Optional;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.discussion.dm.util.ResourceTypeConstants.SOCIAL;

@Service
@Transactional
@RequiredArgsConstructor
public class DiscussionServiceImpl implements DiscussionService {

    private static final String RESOURCE_ID_TYPE_BUSINESS_KEY = "{resourceId: %s, resourceType: %s}";

    private final DiscussionRepository discussionRepository;
    private final ThreadRepository threadRepository;
    private final DiscussionReactionRepository discussionReactionRepository;
    private final ThreadReactionRepository threadReactionRepository;
    private final DiscussionMapper discussionMapper;
    private final ThreadMapper threadMapper;

    @Override
    @Transactional(readOnly = true)
    public DiscussionDTO getDiscussionForResource(String resourceId, String resourceType) {
        UUID resourceIdUuid = UUID.fromString(resourceId);
        Discussion discussion = discussionRepository.findByResourceIdAndResourceType(resourceIdUuid, resourceType)
                .orElseThrow(() -> new WP2ResourceNotFoundException(
                        String.format("Discussion not found for resource: %s, %s", resourceId, resourceType)));
        return discussionMapper.toDTO(discussion);
    }

    @Override
    @Transactional(readOnly = true)
    public DiscussionDTO getDiscussionById(String id, boolean includeThreads) {
        UUID uuid = UUID.fromString(id);
        Discussion discussion = discussionRepository.findById(uuid)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Discussion not found with id: %s", id)));
        return includeThreads ? discussionMapper.toDTO(discussion) : discussionMapper.toDTOWithoutThreads(discussion);
    }

    @Override
    @Transactional(readOnly = true)
    public Page<DiscussionDTO> getDiscussions(Pageable pageable, DiscussionFilterRequest filterRequest, boolean includeThreads) {

        Page<Discussion> discussions;

        if (filterRequest != null && filterRequest.hasAnyFilter()) {
            Specification<Discussion> filterSpec = DiscussionSpecifications
                    .hasVisibility(filterRequest.visibility())
                    .and(DiscussionSpecifications.hasResourceType(filterRequest.resourceType()))
                    .and(DiscussionSpecifications.isCreatedAfter(filterRequest.createdAfter()));
            discussions = discussionRepository.findAll(filterSpec, pageable);
        } else {
            discussions = discussionRepository.findAll(pageable);
        }

        if (includeThreads) {
            return discussions.map(discussionMapper::toDTO);
        } else {
            return discussions.map(discussionMapper::toDTOWithoutThreads);
        }
    }

    @Override
    public DiscussionDTO createDiscussion(DiscussionDTO discussionDTO) {

        if (discussionDTO.getResourceId() != null &&
                discussionRepository.existsByResourceIdAndResourceType(discussionDTO.getResourceId(), discussionDTO.getResourceType())) {
            throw new WP2DuplicateResourceException(
                    RESOURCE_ID_TYPE_BUSINESS_KEY.formatted(discussionDTO.getResourceId(), discussionDTO.getResourceType()));
        }

        Discussion discussion = discussionMapper.toEntity(discussionDTO);
        Discussion savedDiscussion = discussionRepository.saveAndFlush(discussion);
        return discussionMapper.toDTOWithoutThreads(savedDiscussion);
    }

    @Override
    public ResponseEntity<Void> deleteDiscussion(String id) {
        UUID uuid = UUID.fromString(id);

        if (!discussionRepository.existsById(uuid)) {
            throw new WP2ResourceNotFoundException(String.format("Discussion not found with id: %s", id));
        }

        discussionRepository.deleteById(uuid);
        return ResponseEntity.noContent().build();
    }

    @Override
    @Transactional(readOnly = true)
    public Page<ThreadDTO> getThreadsByDiscussionId(String discussionId, Pageable pageable) {
        UUID uuid = UUID.fromString(discussionId);
        Discussion discussion = discussionRepository.findById(uuid)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Discussion not found with id: %s", discussionId)));
        Page<Thread> threads = threadRepository.findByDiscussionAndParentIsNull(discussion, pageable);
        return threads.map(threadMapper::toDTO);
    }

    @Override
    public ResponseEntity<ThreadDTO> addThread(String discussionId, ThreadDTO threadDTO) {
        UUID discussionUuid = UUID.fromString(discussionId);
        Discussion discussion = discussionRepository.findById(discussionUuid)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Discussion not found with id: %s", discussionId)));

        Thread thread = Thread.builder()
                .discussion(discussion)
                .createdBy(threadDTO.getCreatedBy())
                .content(threadDTO.getContent())
                .createdAt(Instant.now())
                .build();

        // Handle parent thread if parentId is provided
        if (threadDTO.getParentId() != null) {
            threadRepository.findById(threadDTO.getParentId())
                    .ifPresent(thread::setParent);
        }
        Thread threadEntity = threadRepository.saveAndFlush(thread);
        return ResponseEntity.status(HttpStatus.CREATED).body(threadMapper.toDTO(threadEntity));
    }

    @Override
    public ResponseEntity<Void> deleteThread(String discussionId, String threadId) {
        UUID discussionUuid = UUID.fromString(discussionId);
        UUID threadUuid = UUID.fromString(threadId);

        Discussion discussion = discussionRepository.findById(discussionUuid)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Discussion not found with id: %s", discussionId)));

        Thread thread = threadRepository.findById(threadUuid)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Thread not found with id: %s", threadId)));

        if (!thread.getDiscussion().getId().equals(discussion.getId())) {
            return ResponseEntity.badRequest().build();
        }

        threadRepository.deleteById(threadUuid);
        return ResponseEntity.noContent().build();
    }

    @Override
    public ResponseEntity<ThreadDTO> updateThread(String discussionId, String threadId, ThreadDTO threadDTO) {
        UUID discussionUuid = UUID.fromString(discussionId);
        UUID threadUuid = UUID.fromString(threadId);

        Discussion discussion = discussionRepository.findById(discussionUuid)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Discussion not found with id: %s", discussionId)));

        Thread thread = threadRepository.findById(threadUuid)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Thread not found with id: %s", threadId)));

        // Verify that the thread belongs to the specified discussion
        if (!thread.getDiscussion().getId().equals(discussionUuid)) {
            return ResponseEntity.badRequest().build();
        }

        // Update allowed fields (currently only content)
        if (threadDTO.getContent() != null) {
            thread.setContent(threadDTO.getContent());
            thread.setModifiedAt(Instant.now());
        }
        Thread threadEntity = threadRepository.saveAndFlush(thread);

        return ResponseEntity.ok(threadMapper.toDTO(threadEntity));
    }

    @Override
    public ResponseEntity<Void> addReactionToDiscussion(String discussionId, String userId, ReactionType reactionType) {
        UUID discussionUuid = UUID.fromString(discussionId);

        Discussion discussion = discussionRepository.findById(discussionUuid)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Discussion not found with id: %s", discussionId)));

        // Check if reaction already exists
        Optional<DiscussionReaction> existingReaction =
                discussionReactionRepository.findByDiscussionAndUserId(discussion, userId);

        if (existingReaction.isEmpty()) {
            DiscussionReaction reaction = DiscussionReaction.builder()
                    .discussion(discussion)
                    .userId(userId)
                    .reaction(reactionType)
                    .createdAt(Instant.now())
                    .build();

            discussionReactionRepository.saveAndFlush(reaction);
            return ResponseEntity.status(HttpStatus.CREATED).build();
        } else {
            if (!existingReaction.get().getReaction().equals(reactionType)) {
                existingReaction.get().setReaction(reactionType);
                discussionReactionRepository.saveAndFlush(existingReaction.get());
                return ResponseEntity.status(HttpStatus.CREATED).build();
            }
            discussionReactionRepository.delete(existingReaction.get());
            return ResponseEntity.status(HttpStatus.CREATED).build();

        }
    }

    @Override
    public ResponseEntity<Void> removeReactionFromDiscussion(String discussionId, String userId) {
        UUID discussionUuid = UUID.fromString(discussionId);

        Discussion discussion = discussionRepository.findById(discussionUuid)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Discussion not found with id: %s", discussionId)));

        discussionReactionRepository.deleteByDiscussionAndUserId(discussion, userId);
        return ResponseEntity.noContent().build();
    }

    @Override
    public ResponseEntity<Void> addReactionToThread(String discussionId, String threadId, String userId, ReactionType reactionType) {
        UUID discussionUuid = UUID.fromString(discussionId);
        UUID threadUuid = UUID.fromString(threadId);

        Discussion discussion = discussionRepository.findById(discussionUuid)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Discussion not found with id: %s", discussionId)));

        Thread thread = threadRepository.findById(threadUuid)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Thread not found with id: %s", threadId)));

        if (!thread.getDiscussion().getId().equals(discussion.getId())) {
            return ResponseEntity.badRequest().build();
        }

        Optional<ThreadReaction> existingReaction =
                threadReactionRepository.findByThreadAndUserId(thread, userId);

        if (existingReaction.isEmpty()) {
            ThreadReaction reaction = ThreadReaction.builder()
                    .thread(thread)
                    .userId(userId)
                    .reaction(reactionType)
                    .createdAt(Instant.now())
                    .build();

            threadReactionRepository.saveAndFlush(reaction);
            return ResponseEntity.status(HttpStatus.CREATED).build();
        } else {

            if (!existingReaction.get().getReaction().equals(reactionType)) {
                existingReaction.get().setReaction(reactionType);
                threadReactionRepository.saveAndFlush(existingReaction.get());
                return ResponseEntity.status(HttpStatus.CREATED).build();
            }

            threadReactionRepository.delete(existingReaction.get());
            return ResponseEntity.status(HttpStatus.CREATED).build();
        }
    }

    @Override
    public ResponseEntity<Void> removeReactionFromThread(String discussionId, String threadId, String userId) {
        UUID discussionUuid = UUID.fromString(discussionId);
        UUID threadUuid = UUID.fromString(threadId);

        Discussion discussion = discussionRepository.findById(discussionUuid)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Discussion not found with id: %s", discussionId)));

        Thread thread = threadRepository.findById(threadUuid)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Thread not found with id: %s", threadId)));

        // Verify that the thread belongs to the specified discussion
        if (!thread.getDiscussion().getId().equals(discussionUuid)) {
            return ResponseEntity.badRequest().build();
        }

        threadReactionRepository.deleteByThreadAndUserId(thread, userId);
        return ResponseEntity.noContent().build();
    }

    @Override
    public DiscussionDTO updateDiscussion(UUID discussionId, InputUpdateDiscussionDto inputUpdateDiscussionDto, boolean includeThreads, String currentUser) {
        // Retrieve the discussion from the repository
        Discussion discussion = discussionRepository.findById(discussionId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(discussionId));

        if (!SOCIAL.equalsIgnoreCase(discussion.getResourceType())) {
            throw new WP2AuthorizationException("Only social discussions can be updated.");
        }

        if (!discussion.getCreatedBy().equals(currentUser)) {
            throw new WP2AuthorizationException("You are not authorized to update this discussion");
        }

        // Update fields if provided
        if (inputUpdateDiscussionDto.getTitle() != null && !inputUpdateDiscussionDto.getTitle().isBlank()) {
            discussion.setTitle(inputUpdateDiscussionDto.getTitle());
        }

        if (inputUpdateDiscussionDto.getContent() != null && !inputUpdateDiscussionDto.getContent().isBlank()) {
            discussion.setContent(inputUpdateDiscussionDto.getContent());
        }

        if (inputUpdateDiscussionDto.getUrl() != null) {
            discussion.setUrl(inputUpdateDiscussionDto.getUrl());
        }

        discussion.setUpdatedBy(currentUser);

        Discussion updatedDiscussion = discussionRepository.saveAndFlush(discussion);

        return includeThreads ? discussionMapper.toDTO(updatedDiscussion) : discussionMapper.toDTOWithoutThreads(updatedDiscussion);
    }

}
