package com.finconsgroup.itserr.marketplace.discussion.dm.repository.specification;

import com.finconsgroup.itserr.marketplace.discussion.dm.entity.Discussion;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.lang.NonNull;

import java.time.Instant;

/**
 * {@link com.finconsgroup.itserr.marketplace.discussion.dm.entity.Discussion} specifications
 */
public final class DiscussionSpecifications {

    private static final String VISIBILITY = "visibility";
    private static final String RESOURCE_TYPE = "resourceType";
    private static final String CREATED_AT = "createdAt";

    /**
     * Private constructor.
     */
    private DiscussionSpecifications() {
        throw new UnsupportedOperationException("Cannot be instantiated");
    }

    /**
     * Creates a {@link Specification} to find {@link Discussion} by visibility.
     *
     * @param visibility the visibility to filter, or null.
     * @return A {@link Specification} matching visibility.
     */
    @NonNull
    public static Specification<Discussion> hasVisibility(
            String visibility) {
        return (root, query, cb) ->
                StringUtils.isBlank(visibility)
                        ? cb.conjunction()
                        : cb.equal(root.get(VISIBILITY), visibility);
    }

    /**
     * Creates a {@link Specification} to find {@link Discussion} by resourceType.
     *
     * @param resourceType the resourceType to filter, or null.
     * @return A {@link Specification} matching resourceType.
     */
    @NonNull
    public static Specification<Discussion> hasResourceType(
            String resourceType) {
        return (root, query, cb) ->
                StringUtils.isBlank(resourceType)
                        ? cb.conjunction()
                        : cb.equal(root.get(RESOURCE_TYPE), resourceType);
    }

    /**
     * Creates a {@link Specification} to find {@link Discussion} by resourceType.
     *
     * @param createdAfter filter to return discussions created after, or null.
     * @return A {@link Specification} matching resourceType.
     */
    @NonNull
    public static Specification<Discussion> isCreatedAfter(
            Instant createdAfter) {
        return (root, query, cb) ->
                createdAfter == null
                        ? cb.conjunction()
                        : cb.greaterThanOrEqualTo(root.get(CREATED_AT), createdAfter);
    }
}

