package com.finconsgroup.itserr.marketplace.discussion.bs.client.dm;

import com.finconsgroup.itserr.marketplace.discussion.bs.dto.DiscussionDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.InputUpdateDiscussionDto;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.ThreadDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.enums.ReactionType;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import java.time.Instant;

/**
 * Client interface for interacting with the Discussion DM service.
 *
 * <p>This client is expected to be implemented via HTTP (e.g., RestTemplate/WebClient/OpenFeign)
 * by the core-web module. It exposes minimal operations used by the BS layer.</p>
 */

public interface DiscussionDmClient {

    /**
     * Retrieves a discussion by its unique identifier.
     *
     * @param id the unique identifier of the discussion
     * @param includeThreads whether to include threads in the response (default: true)
     * @return the {@link DiscussionDTO} representing the discussion
     */
    @GetMapping("/discussion/{id}")
    DiscussionDTO getDiscussionById(@PathVariable String id,
                                    @RequestParam(defaultValue = "true", required = false) boolean includeThreads);

    /**
     * Creates a new discussion based on the provided discussion data.
     *
     * @param discussionDTO an object containing the details of the discussion to be created
     * @return a response entity with no content, indicating the outcome of the discussion creation
     */
    @PostMapping("/discussion")
    DiscussionDTO createDiscussion(@RequestBody DiscussionDTO discussionDTO);

    /**
     * Updates an existing discussion with the provided details.
     * This method allows for partial updates to specific fields of a discussion.
     *
     * @param discussionId the unique identifier of the discussion to update
     * @param updatedDiscussionDTO an object containing the fields to be updated in the discussion
     * @param includeThreads whether to include threads in the response (default: true)
     * @return a response entity containing the updated discussion details
     */
    @PatchMapping("/discussion/{discussionId}")
    DiscussionDTO updateDiscussion(
            @PathVariable String discussionId,
            @RequestBody InputUpdateDiscussionDto updatedDiscussionDTO,
            @RequestParam(defaultValue = "true", required = false) boolean includeThreads
    );

    /**
     * Retrieves a paginated list of discussions from DM service.
     *
     * @param page           page number (0-based)
     * @param size           page size
     * @param sortBy         sorting field
     * @param direction      sort direction (asc|desc)
     * @param visibility     optional visibility filter
     * @param includeThreads whether to include threads in the response
     * @param resourceType optional filter by the type of the resource
     * @param createdAfter optional filter for created after
     * @return page of discussions
     */
    @GetMapping("/discussions")
    Page<DiscussionDTO> getDiscussions(
            @RequestParam int page,
            @RequestParam int size,
            @RequestParam(required = false) String sortBy,
            @RequestParam(required = false) String direction,
            @RequestParam(required = false) String visibility,
            @RequestParam(defaultValue = "false") boolean includeThreads,
            @RequestParam(required = false) String resourceType,
            @RequestParam(required = false) Instant createdAfter
    );

    @GetMapping("/discussion/{discussionId}/threads")
    Page<ThreadDTO> getThreadsByDiscussionId(@PathVariable String discussionId,
                                             @RequestParam(defaultValue = "0") int page,
                                             @RequestParam(defaultValue = "10") int size,
                                             @RequestParam(defaultValue = "createdAt") String sortBy,
                                             @RequestParam(defaultValue = "DESC") Sort.Direction direction);

    /**
     * Deletes a discussion by its identifier.
     *
     * @param id the unique identifier of the discussion to delete
     * @return a response entity indicating the outcome (typically 204 No Content)
     */
    @DeleteMapping("/discussion/{id}")
    ResponseEntity<Void> deleteDiscussion(@PathVariable String id);

    /**
     * Adds a new thread to the specified discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param threadDTO    the thread payload to create
     * @return a response entity indicating the outcome (typically 201 Created)
     */
    @PostMapping("/discussion/{discussionId}/threads")
    ResponseEntity<ThreadDTO> addThread(@PathVariable String discussionId, @RequestBody ThreadDTO threadDTO);

    /**
     * Deletes a specific thread from a discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param threadId     the unique identifier of the thread to delete
     * @return a response entity indicating the outcome (typically 204 No Content)
     */
    @DeleteMapping("/discussion/{discussionId}/threads/{threadId}")
    ResponseEntity<Void> deleteThread(@PathVariable String discussionId, @PathVariable String threadId);

    /**
     * Updates a specific thread within a discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param threadId     the unique identifier of the thread to update
     * @param threadDTO    the updated thread payload
     * @return a response entity indicating the outcome (e.g., 200 OK or 204 No Content)
     */
    @PutMapping("/discussion/{discussionId}/threads/{threadId}")
    ResponseEntity<ThreadDTO> updateThread(@PathVariable String discussionId, @PathVariable String threadId, @RequestBody ThreadDTO threadDTO);

    /**
     * Adds a reaction to a discussion by a user.
     *
     * @param discussionId the unique identifier of the discussion
     * @param userId       the unique identifier of the user adding the reaction
     * @param reactionType the type of reaction to add
     * @return a response entity indicating the outcome (e.g., 200 OK or 201 Created)
     */
    @PostMapping("/discussion/{discussionId}/reactions/{userId}")
    ResponseEntity<Void> addReactionToDiscussion(@PathVariable String discussionId, @PathVariable String userId, @RequestParam ReactionType reactionType);

    /**
     * Removes a user's reaction from a discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param userId       the unique identifier of the user
     * @return a response entity indicating the outcome (typically 204 No Content)
     */
    @DeleteMapping("/discussion/{discussionId}/reactions/{userId}")
    ResponseEntity<Void> removeReactionFromDiscussion(@PathVariable String discussionId, @PathVariable String userId);

    /**
     * Adds a reaction to a specific thread within a discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param threadId     the unique identifier of the thread
     * @param userId       the unique identifier of the user adding the reaction
     * @param reactionType the type of reaction to add
     * @return a response entity indicating the outcome (e.g., 200 OK or 201 Created)
     */
    @PostMapping("/discussion/{discussionId}/threads/{threadId}/reactions/{userId}")
    ResponseEntity<Void> addReactionToThread(@PathVariable String discussionId, @PathVariable String threadId,
                                             @PathVariable String userId, @RequestParam ReactionType reactionType);

    /**
     * Removes a user's reaction from a specific thread within a discussion.
     *
     * @param discussionId the unique identifier of the discussion
     * @param threadId     the unique identifier of the thread
     * @param userId       the unique identifier of the user
     * @return a response entity indicating the outcome (typically 204 No Content)
     */
    @DeleteMapping("/discussion/{discussionId}/threads/{threadId}/reactions/{userId}")
    ResponseEntity<Void> removeReactionFromThread(@PathVariable String discussionId, @PathVariable String threadId, @PathVariable String userId);

    /**
     * Retrieves a discussion associated with the specified resource identifier and type.
     *
     * @param resourceId the unique identifier of the resource
     * @param resourceType the type of the resource
     * @return the {@link DiscussionDTO} representing the discussion
     */
    @GetMapping("/discussion/resource/{resourceId}")
    DiscussionDTO getDiscussionForResource(@PathVariable String resourceId, @RequestParam String resourceType);

    @GetMapping("/discussion/{discussionId}")
    DiscussionDTO getDiscussionByDiscussionId(@PathVariable String discussionId);
}
