package com.finconsgroup.itserr.marketplace.discussion.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.discussion.bs.bean.DiscussionApplicationEvent;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.DiscussionDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.service.AdminService;
import com.finconsgroup.itserr.marketplace.discussion.bs.service.DiscussionService;
import jakarta.annotation.PreDestroy;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import static com.finconsgroup.itserr.marketplace.discussion.bs.enums.MessagingEventType.UPDATED;

/**
 * Default implementation of {@link AdminService}
 * to perform admin operations related to Discussion resources.
 */
@Service
@Slf4j
public class DefaultAdminService implements AdminService {

    private final DiscussionService discussionService;
    private final ApplicationEventPublisher applicationEventPublisher;
    private final ExecutorService executor;

    public DefaultAdminService(DiscussionService discussionService,
                               ApplicationEventPublisher applicationEventPublisher) {
        this.discussionService = discussionService;
        this.applicationEventPublisher = applicationEventPublisher;
        this.executor = Executors.newVirtualThreadPerTaskExecutor();
    }


    @Override
    public void publishAllMessages(String resourceType, String visibility) {
        executor.submit(() -> publishAllMessagesAsync(resourceType, visibility));
    }

    @PreDestroy
    void close() {
        executor.shutdown();
    }

    private void publishAllMessagesAsync(String resourceType, String visibility) {
        int pageNumber = 0;
        long fetchedCount = 0;
        long totalCount;
        while (true) {
            Page<DiscussionDTO> discussionsPage = discussionService.getDiscussions(
                    pageNumber, 50, "id", SortDirection.ASC.name(), visibility, false);

            if (discussionsPage.getContent().isEmpty()) {
                break;
            }

            discussionsPage.getContent().forEach(discussionDTO ->
                    applicationEventPublisher.publishEvent(new DiscussionApplicationEvent(discussionDTO, UPDATED))
            );
            totalCount = discussionsPage.getTotalElements();
            fetchedCount += discussionsPage.getContent().size();
            log.info("Published {}/{} discussions", fetchedCount, discussionsPage.getTotalElements());
            if (totalCount <= fetchedCount) {
                break;
            }

            pageNumber++;
        }
    }
}
