package com.finconsgroup.itserr.marketplace.audit.dm.dto;

import io.swagger.v3.oas.annotations.media.Schema;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotBlank;
import jakarta.validation.constraints.NotNull;
import jakarta.validation.constraints.Size;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.util.Map;

import static com.finconsgroup.itserr.marketplace.audit.dm.util.DomainConstants.LONG_TEXT_LENGTH;
import static com.finconsgroup.itserr.marketplace.audit.dm.util.DomainConstants.SHORT_TEXT_LENGTH;

/**
 * Input DTO for creating audit log.
 */
@Schema(name = "InputCreateAuditLogDto", description = "DTO for creating audit log")
@Data
@SuperBuilder
@NoArgsConstructor
@AllArgsConstructor
public class InputCreateAuditLogDto {

    @Schema(description = "The source of the event, usually this represents the microservice which generated the event.",
            example = "catalog-bs")
    @NotBlank
    @Size(max = SHORT_TEXT_LENGTH)
    private String eventSource;

    @Schema(description = "The type of the event.", example = "catalog-item-event-created")
    @NotBlank
    @Size(max = SHORT_TEXT_LENGTH)
    private String eventType;

    @Valid
    @NotNull
    @Schema(description = "The user from User Profile whose action triggered the event.")
    private InputAuditLogUserDto user;

    @Valid
    @NotNull
    @Schema(description = "The resource on which the event was triggered.")
    private InputAuditLogResourceDto resource;

    @Schema(description = "The message indicating the reason for an action.",
            example = "Approved as everything looks correct")
    @Size(max = LONG_TEXT_LENGTH)
    private String eventMessage;

    @Schema(description = "The additional data that was provided on the event by the source microservice",
            example = "{\"notifyUserIds\": [\"a18b4e8f-36f2-4105-8e0e-ba0c0d7dd635\"]}")
    private Map<String, Object> eventAdditionalData;

    @NotNull
    @Schema(description = "The time at which the event action was actually triggered by the user.",
            example = "2025-08-25T11:22:33.123456Z")
    private Instant eventActionTime;

    @NotNull
    @Schema(description = "The time at which the event was generated by the source microservice.",
            example = "2025-08-25T11:23:32.123456Z")
    private Instant eventCreationTime;
}
