package org.finconsgroup.itserr.criterion.security.service;

import org.finconsgroup.itserr.criterion.security.config.ExternalAuthProperties;
import org.finconsgroup.itserr.criterion.security.dto.AuthorizationRequest;
import io.github.resilience4j.circuitbreaker.CallNotPermittedException;
import io.github.resilience4j.circuitbreaker.CircuitBreaker;
import io.github.resilience4j.retry.Retry;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

import java.util.function.Supplier;

/**
 * Service for external authorization verification.
 * Forwards the original Authorization header from the client request
 * to the external auth service.
 */
@Slf4j
public class ExternalAuthService {

    private final RestTemplate restTemplate;
    private final ExternalAuthProperties properties;
    private final CircuitBreaker circuitBreaker;
    private final Retry retry;

    public ExternalAuthService(RestTemplate restTemplate,
                               ExternalAuthProperties properties,
                               CircuitBreaker circuitBreaker,
                               Retry retry) {
        this.restTemplate = restTemplate;
        this.properties = properties;
        this.circuitBreaker = circuitBreaker;
        this.retry = retry;
    }

    /**
     * Check authorization by calling the external service.
     * Forwards the original Authorization header from the client.
     *
     * @param authUrl custom URL (if null/empty, uses default from config)
     * @param authRequest request data containing the original Authorization header
     * @return true if authorized, false otherwise
     */
    public boolean checkAuthorization(String authUrl, AuthorizationRequest authRequest) {
        String url = (authUrl == null || authUrl.isEmpty()) ? properties.getUrl() : authUrl;

        log.debug("Checking authorization at: {}", url);

        // Check if Authorization header is present
        if (authRequest.getAuthorizationHeader() == null || authRequest.getAuthorizationHeader().isEmpty()) {
            log.warn("No Authorization header present in request");
            return false;
        }

        Supplier<Boolean> authCall = () -> executeAuthCall(url, authRequest);

        // Apply Retry if enabled
        if (properties.getRetry().isEnabled()) {
            authCall = Retry.decorateSupplier(retry, authCall);
        }

        // Apply Circuit Breaker if enabled
        if (properties.getCircuitBreaker().isEnabled()) {
            authCall = CircuitBreaker.decorateSupplier(circuitBreaker, authCall);
        }

        try {
            return authCall.get();
        } catch (CallNotPermittedException e) {
            log.warn("Circuit breaker is OPEN. Auth service appears to be unavailable.");
            return handleFallback("Circuit breaker open");
        } catch (Exception e) {
            log.error("Authorization check failed: {}", e.getMessage());
            return handleFallback("Authorization check failed: " + e.getMessage());
        }
    }

    /**
     * Execute the actual HTTP call to the authorization service.
     * Forwards the original Authorization header from the client.
     */
    private Boolean executeAuthCall(String url, AuthorizationRequest authRequest) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        // Forward the original Authorization header from the client
        headers.set(HttpHeaders.AUTHORIZATION, authRequest.getAuthorizationHeader());
        log.debug("Forwarding Authorization header to auth service");

        HttpEntity<AuthorizationRequest> requestEntity = new HttpEntity<>(authRequest, headers);

        ResponseEntity<Boolean> response = restTemplate.postForEntity(
                url,
                requestEntity,
                Boolean.class
        );

        Boolean authorized = response.getBody();

        if (authorized == null) {
            log.warn("Auth service returned null response, treating as unauthorized");
            return false;
        }

        log.debug("Authorization response: {}", authorized);
        return authorized;
    }

    private boolean handleFallback(String reason) {
        if (properties.isFailOpen()) {
            log.warn("Fail-open mode: allowing access despite auth failure. Reason: {}", reason);
            return true;
        } else {
            log.warn("Fail-close mode: denying access due to auth failure. Reason: {}", reason);
            return false;
        }
    }

    public CircuitBreaker.State getCircuitBreakerState() {
        return circuitBreaker.getState();
    }

    public CircuitBreaker.Metrics getCircuitBreakerMetrics() {
        return circuitBreaker.getMetrics();
    }
}