package org.finconsgroup.itserr.criterion.security.config;

import lombok.Data;
import org.springframework.boot.context.properties.ConfigurationProperties;

import java.time.Duration;

@Data
@ConfigurationProperties(prefix = "wp3.external-auth")
public class ExternalAuthProperties {

    private boolean enabled = true;
    private String url = "http://localhost:8081/api/authorize";
    private Duration connectTimeout = Duration.ofSeconds(5);
    private Duration readTimeout = Duration.ofSeconds(5);
    private boolean failOpen = false;

    /**
     * Internal calls configuration.
     * When a request contains the X-Internal-Auth header with the correct secret,
     * the external auth check is skipped.
     */
    private InternalAuthProperties internal = new InternalAuthProperties();

    private RetryProperties retry = new RetryProperties();
    private CircuitBreakerProperties circuitBreaker = new CircuitBreakerProperties();

    @Data
    public static class InternalAuthProperties {

        /**
         * Enable/disable internal auth bypass.
         */
        private boolean enabled = true;

        /**
         * Header name for internal authentication.
         */
        private String headerName = "X-Internal-Auth";

        /**
         * Shared secret for internal calls.
         * MUST be the same across all microservices!
         */
        private String secret;

        /**
         * Check if internal auth is properly configured.
         */
        public boolean isConfigured() {
            return enabled && secret != null && !secret.isEmpty();
        }
    }

    @Data
    public static class RetryProperties {
        private boolean enabled = true;
        private int maxAttempts = 3;
        private Duration waitDuration = Duration.ofSeconds(1);
        private double multiplier = 2.0;
        private Duration maxWaitDuration = Duration.ofSeconds(10);
    }

    @Data
    public static class CircuitBreakerProperties {
        private boolean enabled = true;
        private float failureRateThreshold = 50;
        private Duration waitDurationInOpenState = Duration.ofSeconds(30);
        private int permittedNumberOfCallsInHalfOpenState = 3;
        private int slidingWindowSize = 10;
        private int minimumNumberOfCalls = 5;
    }
}