package org.finconsgroup.itserr.criterion.common.http.config;

import lombok.extern.slf4j.Slf4j;
import org.finconsgroup.itserr.criterion.common.http.interceptor.HeaderPropagatingInterceptor;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.context.annotation.Bean;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.web.client.RestTemplate;

@AutoConfiguration
@EnableConfigurationProperties(org.finconsgroup.itserr.criterion.common.http.config.PropagatingRestTemplateProperties.class)
@ConditionalOnProperty(prefix = "wp3.http-client", name = "enabled", havingValue = "true", matchIfMissing = true)
@Slf4j
public class PropagatingRestTemplateAutoConfiguration {

    @Bean
    @ConditionalOnMissingBean(name = "internalRestTemplate")
    public RestTemplate internalRestTemplate(
            ObjectProvider<RestTemplateBuilder> builderProvider,
            org.finconsgroup.itserr.criterion.common.http.config.PropagatingRestTemplateProperties properties) {

        log.info("Creating InternalRestTemplate with header propagation and internal auth");

        RestTemplate restTemplate;

        RestTemplateBuilder builder = builderProvider.getIfAvailable();
        if (builder != null) {
            restTemplate = builder
                    .connectTimeout(properties.getConnectTimeout())
                    .readTimeout(properties.getReadTimeout())
                    .build();
        } else {
            SimpleClientHttpRequestFactory factory = new SimpleClientHttpRequestFactory();
            factory.setConnectTimeout((int) properties.getConnectTimeout().toMillis());
            factory.setReadTimeout((int) properties.getReadTimeout().toMillis());
            restTemplate = new RestTemplate(factory);
        }

        // Add interceptor
        restTemplate.getInterceptors().add(new HeaderPropagatingInterceptor(properties));

        return restTemplate;
    }
}